import argparse
import logging
import os
import django

os.environ.setdefault(
    "DJANGO_SETTINGS_MODULE", "cleargrid.mc3_integration.settings"
)
django.setup()


from cleargrid.logging import add_log_file  # noqa
from cleargrid.mc3_integration.runner import Runner  # noqa
from cleargrid.mc3_integration.updater import Updater  # noqa


FORMAT = (
    "%(asctime)s - %(filename)12s:%(lineno)4d - "
    "%(threadName)10s[%(levelname)7s]:%(message)s"
)


class StdoutUpdater(Updater):
    def post(self, *args, **kwargs):
        print(f"POST: {args[0]}")
        for k, v in kwargs.items():
            print(f"\t{k:10s}:{v}")


class NullUpdater(Updater):
    def post(self, *args, **kwargs):
        pass


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--stdout",
        action="store_true",
        help="Print updates to STDOUT rather than posting to server",
    )
    parser.add_argument(
        "--null", action="store_true", help="suppress updater output entirely"
    )
    parser.add_argument(
        "-L",
        "--log-level",
        dest="log_level",
        type=str,
        default="INFO",
        help="Log level to set logger to",
    )
    parser.add_argument(
        "--log-file",
        dest="log_file",
        type=str,
        default="True",
        help="File to log to",
    )
    args = parser.parse_args()

    logging.basicConfig(
        format=FORMAT, level=getattr(logging, args.log_level.upper(), "INFO"),
    )
    add_log_file(args.log_file, "mc3_integrations")

    p = {}
    if args.stdout:
        p.update(updater=StdoutUpdater())
    if args.null:
        p.update(updater=NullUpdater())

    runner = Runner(**p)
    try:
        runner.start()
        runner.join()
    except KeyboardInterrupt:
        logging.info("Received CTRL-C, clean shutdown")
    finally:
        runner.stop()
        runner.join()


if __name__ == "__main__":
    main()
