import sys
import uuid

try:
    from django.db import models
except Exception:
    print("There was an error loading django modules. Do you have django installed?")
    sys.exit()


class UniqueIdentifierField(models.UUIDField):
    """
    pyODBC does not support unique identifier field natively. 
    See https://github.com/michiya/django-pyodbc-azure/issues/79
    """

    def db_type(self, connection):
        if self.primary_key:
            return "uniqueidentifier default (newsequentialid())"
        else:
            return "uniqueidentifier"

    def rel_db_type(self, connection):
        return "uniqueidentifier"

    # leave id out of payload on insert
    def contribute_to_class(self, cls, name, **kwargs):
        assert not self.primary_key or (
            self.primary_key and not cls._meta.auto_field
        ), "A model can't have more than one AutoField."
        super().contribute_to_class(cls, name, **kwargs)
        if self.primary_key:
            cls._meta.auto_field = self

    def get_db_prep_value(self, value, connection, prepared=False):
        if value is None:
            return None
        if not isinstance(value, uuid.UUID):
            value = self.to_python(value)

        return str(value)

    def from_db_value(self, value, expression, connection):
        return self._to_uuid(value)

    def to_python(self, value):
        return self._to_uuid(value)

    def _to_uuid(self, value):
        if value is not None and not isinstance(value, uuid.UUID):
            try:
                return uuid.UUID(value)
            except (AttributeError, ValueError):
                print("An error occured")
                # raise exceptions.ValidationError(
                #     self.error_messages['invalid'],
                #     code='invalid',
                #     params={'value': value},
                # )
        return value


class Endpoint(models.Model):
    PK = models.CharField(
        db_column="PK", default=uuid.uuid4, primary_key=True, max_length=36
    )  # Field name made lowercase.
    meter = models.ForeignKey(
        "Meter", models.DO_NOTHING, default=uuid.uuid4, db_column="MeterPK", related_name="endpoint"
    )  # Field name made lowercase.
    # UniqueIdentifierField(primary_key=False, max_length=36, editable=False, default=None,  db_column='MeterPK')
    identifier = models.CharField(
        db_column="Identifier", max_length=18
    )  # Field name made lowercase.
    macaddress = models.CharField(
        db_column="MACAddress", max_length=12
    )  # Field name made lowercase.
    latitude = models.CharField(db_column="Latitude", max_length=17)  # Field name made lowercase.
    longitude = models.CharField(db_column="Longitude", max_length=17)  # Field name made lowercase.
    type = models.CharField(db_column="Type", max_length=32)  # Field name made lowercase.
    scmerttype = models.SmallIntegerField(db_column="SCMErtType")  # Field name made lowercase.
    deviceclass = models.SmallIntegerField(db_column="DeviceClass")  # Field name made lowercase.
    tamper1count = models.SmallIntegerField(db_column="Tamper1Count")  # Field name made lowercase.
    tamper2count = models.SmallIntegerField(db_column="Tamper2Count")  # Field name made lowercase.
    previoustamper1count = models.SmallIntegerField(
        db_column="PreviousTamper1Count"
    )  # Field name made lowercase.
    previoustamper2count = models.SmallIntegerField(
        db_column="PreviousTamper2Count"
    )  # Field name made lowercase.

    def __eq__(self, other):
        if self.__class__ == other.__class__:
            fields = [field.name for field in self._meta.fields]
            for field in fields:
                if not getattr(self, field) == getattr(other, field):
                    return False
            return True
        else:
            raise TypeError("Comparing object is not of the same type.")

    class Meta:
        managed = True
        db_table = "EndPoint"


class Endpointlocation(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    endpointidentifier = models.CharField(
        db_column="EndPointIdentifier", max_length=18
    )  # Field name made lowercase.
    markettype = models.CharField(
        db_column="MarketType", max_length=1, blank=True, null=True
    )  # Field name made lowercase.
    uniqueid = models.CharField(
        db_column="UniqueID", unique=True, max_length=19, blank=True, null=True
    )  # Field name made lowercase.
    streetaddress = models.CharField(
        db_column="StreetAddress", max_length=255, blank=True, null=True
    )  # Field name made lowercase.
    city = models.CharField(
        db_column="City", max_length=100, blank=True, null=True
    )  # Field name made lowercase.
    region = models.CharField(
        db_column="Region", max_length=100, blank=True, null=True
    )  # Field name made lowercase.
    zip = models.CharField(
        db_column="Zip", max_length=50, blank=True, null=True
    )  # Field name made lowercase.
    country = models.CharField(
        db_column="Country", max_length=100, blank=True, null=True
    )  # Field name made lowercase.
    latitude = models.CharField(
        db_column="Latitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    longitude = models.CharField(
        db_column="Longitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    geopointsource = models.CharField(
        db_column="GeoPointSource", max_length=255, blank=True, null=True
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "EndPointLocation"


class Endpointinformation(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    markettype = models.CharField(
        db_column="MarketType", max_length=1
    )  # Field name made lowercase.
    endpointphase = models.CharField(
        db_column="EndpointPhase", max_length=5
    )  # Field name made lowercase.
    minendpointid = models.CharField(
        db_column="MinEndpointID", max_length=18
    )  # Field name made lowercase.
    maxendpointid = models.CharField(
        db_column="MaxEndpointID", max_length=18
    )  # Field name made lowercase.
    isleaksensor = models.BooleanField(db_column="IsLeakSensor")  # Field name made lowercase.
    deviceclass = models.SmallIntegerField(db_column="DeviceClass")  # Field name made lowercase.
    comments = models.CharField(
        db_column="Comments", max_length=50, blank=True, null=True
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "EndpointInformation"


class Extendeddlreading(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    extdlreq = models.CharField(
        db_column="ExtDLReqPK", max_length=36
    )  # Field name made lowercase.
    timeoccurred = models.DateTimeField(db_column="TimeOccurred")  # Field name made lowercase.
    operatorusername = models.CharField(
        db_column="OperatorUserName", max_length=15
    )  # Field name made lowercase.
    readdata1 = models.TextField(db_column="ReadData1")  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "ExtendedDLReading"


class Extendeddlrequest(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    meter = models.ForeignKey(
        "Meter", models.DO_NOTHING, db_column="MeterPK", related_name="extended_dl_requests"
    )  # Field name made lowercase.
    endpointidentifier = models.CharField(
        db_column="EndPointIdentifier", max_length=18
    )  # Field name made lowercase.
    markettype = models.CharField(
        db_column="MarketType", max_length=1
    )  # Field name made lowercase.
    sequencenumber = models.SmallIntegerField(
        db_column="SequenceNumber"
    )  # Field name made lowercase.
    parentsequencenumber = models.SmallIntegerField(
        db_column="ParentSequenceNumber"
    )  # Field name made lowercase.
    isrequired = models.BooleanField(db_column="IsRequired")  # Field name made lowercase.
    readstatus = models.SmallIntegerField(db_column="ReadStatus")  # Field name made lowercase.
    requestorigin = models.SmallIntegerField(
        db_column="RequestOrigin"
    )  # Field name made lowercase.
    command = models.IntegerField(db_column="Command")  # Field name made lowercase.
    parameters = models.CharField(
        db_column="Parameters", max_length=40
    )  # Field name made lowercase.
    securedrequest1 = models.CharField(
        db_column="SecuredRequest1", max_length=500
    )  # Field name made lowercase.
    securedrequest2 = models.CharField(
        db_column="SecuredRequest2", max_length=4000, blank=True, null=True
    )  # Field name made lowercase.
    ismanual = models.BooleanField(db_column="IsManual")  # Field name made lowercase.
    isextended = models.BooleanField(db_column="IsExtended")  # Field name made lowercase.
    techdescriptor = models.SmallIntegerField(
        db_column="TechDescriptor"
    )  # Field name made lowercase.
    commandinfo = models.CharField(
        db_column="CommandInfo", max_length=128
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "ExtendedDLRequest"


class Extendedreading(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    # meterPK = models.CharField(db_column='MeterPK', max_length=36, unique=True)  # Field name made lowercase.
    meter = UniqueIdentifierField(
        primary_key=False,
        max_length=36,
        editable=False,
        default=None,
        db_column="MeterPK",
    )
    readdatetime = models.DateTimeField(db_column="ReadDateTime")  # Field name made lowercase.
    readstatus = models.SmallIntegerField(db_column="ReadStatus")  # Field name made lowercase.
    sequencenumber = models.SmallIntegerField(
        db_column="SequenceNumber"
    )  # Field name made lowercase.
    readdata = models.CharField(db_column="ReadData", max_length=3600)  # Field name made lowercase.

    def __eq__(self, other):
        if self.__class__ == other.__class__:
            fields = [field.name for field in self._meta.fields]
            for field in fields:
                if not getattr(self, field) == getattr(other, field):
                    return False
            return True
        else:
            raise TypeError("Comparing object is not of the same type.")

    class Meta:
        managed = True
        db_table = "ExtendedReading"


class Lostmeter(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    identifier = models.CharField(
        db_column="Identifier", max_length=18
    )  # Field name made lowercase.
    markettype = models.CharField(
        db_column="MarketType", max_length=1
    )  # Field name made lowercase.
    scmerttype = models.SmallIntegerField(db_column="SCMErtType")  # Field name made lowercase.
    deviceclass = models.SmallIntegerField(db_column="DeviceClass")  # Field name made lowercase.
    estimatedlatitude = models.CharField(
        db_column="EstimatedLatitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    estimatedlongitude = models.CharField(
        db_column="EstimatedLongitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    assignedlatitude = models.CharField(
        db_column="AssignedLatitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    assignedlongitude = models.CharField(
        db_column="AssignedLongitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    state = models.IntegerField(db_column="State")  # Field name made lowercase.
    street = models.CharField(
        db_column="Street", max_length=60, blank=True, null=True
    )  # Field name made lowercase.
    comments = models.CharField(
        db_column="Comments", max_length=300, blank=True, null=True
    )  # Field name made lowercase.
    ismapped = models.BooleanField(db_column="IsMapped")  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "LostMeter"


class Lostmeterreading(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    lostmeterPK = models.CharField(
        db_column="LostMeterPK", max_length=36
    )  # Field name made lowercase.
    timeoccurred = models.DateTimeField(db_column="TimeOccurred")  # Field name made lowercase.
    rssi = models.IntegerField(db_column="RSSI")  # Field name made lowercase.
    latitude = models.CharField(
        db_column="Latitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    longitude = models.CharField(
        db_column="Longitude", max_length=17, blank=True, null=True
    )  # Field name made lowercase.
    packettype = models.SmallIntegerField(db_column="PacketType")  # Field name made lowercase.
    isradioread = models.BooleanField(db_column="IsRadioRead")  # Field name made lowercase.
    isconsumptiveerror = models.BooleanField(
        db_column="IsConsumptiveError"
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "LostMeterReading"


class Message(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    meterPK = models.CharField(db_column="MeterPK", max_length=36)  # Field name made lowercase.
    messageindicator = models.SmallIntegerField(
        db_column="MessageIndicator"
    )  # Field name made lowercase.
    changedmessageindicator = models.BooleanField(
        db_column="ChangedMessageIndicator"
    )  # Field name made lowercase.
    messagetext = models.CharField(
        db_column="MessageText", max_length=160, blank=True, null=True
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "Message"


class Meter(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    routePK = models.ForeignKey(
        "Route", models.DO_NOTHING, db_column="RoutePK", blank=True, null=True
    )  # Field name made lowercase.
    segmentPK = models.ForeignKey(
        "Segment", models.DO_NOTHING, db_column="SegmentPK", blank=True, null=True
    )  # Field name made lowercase.
    street = models.CharField(db_column="Street", max_length=60)  # Field name made lowercase.
    messagecount = models.IntegerField(db_column="MessageCount")  # Field name made lowercase.
    isreadonly = models.BooleanField(db_column="IsReadOnly")  # Field name made lowercase.
    isoutofroute = models.BooleanField(db_column="IsOutofRoute")  # Field name made lowercase.
    isduplicateoor = models.BooleanField(db_column="IsDuplicateOOR")  # Field name made lowercase.
    identifier = models.CharField(
        db_column="Identifier", max_length=18
    )  # Field name made lowercase.
    sequencenumber = models.IntegerField(db_column="SequenceNumber")  # Field name made lowercase.
    symbolcode = models.IntegerField(db_column="SymbolCode")  # Field name made lowercase.
    externalmeterid = models.CharField(
        db_column="ExternalMeterID", max_length=14, blank=True, null=True
    )  # Field name made lowercase.
    dials = models.SmallIntegerField(
        db_column="Dials", blank=True, null=True
    )  # Field name made lowercase.
    decimals = models.SmallIntegerField(
        db_column="Decimals", blank=True, null=True
    )  # Field name made lowercase.
    typeofread = models.CharField(
        db_column="TypeOfRead", max_length=6, blank=True, null=True
    )  # Field name made lowercase.
    markettype = models.CharField(
        db_column="MarketType", max_length=1, blank=True, null=True
    )  # Field name made lowercase.
    readstatus = models.SmallIntegerField(
        db_column="ReadStatus", blank=True, null=True
    )  # Field name made lowercase.
    readcomplete = models.BooleanField(db_column="ReadComplete")  # Field name made lowercase.
    parentid = models.CharField(
        db_column="ParentID", max_length=30, blank=True, null=True
    )  # Field name made lowercase.
    # uniqueid = models.CharField(db_column='UniqueId', max_length=19, blank=True, null=True)  # Field name made lowercase.
    nonamr = models.BooleanField(db_column="NonAMR")  # Field name made lowercase.
    locationcode = models.CharField(
        db_column="LocationCode", max_length=2, blank=True, null=True
    )  # Field name made lowercase.
    readinstructioncode1 = models.CharField(
        db_column="ReadInstructionCode1", max_length=2, blank=True, null=True
    )  # Field name made lowercase.
    readinstructioncode2 = models.CharField(
        db_column="ReadInstructionCode2", max_length=2, blank=True, null=True
    )  # Field name made lowercase.
    information = models.CharField(
        db_column="Information", max_length=20, blank=True, null=True
    )  # Field name made lowercase.
    activeindicator = models.CharField(
        db_column="ActiveIndicator", max_length=1, blank=True, null=True
    )  # Field name made lowercase.
    validationtype = models.SmallIntegerField(
        db_column="ValidationType", blank=True, null=True
    )  # Field name made lowercase.
    previousread = models.BigIntegerField(
        db_column="PreviousRead", blank=True, null=True
    )  # Field name made lowercase.
    high2 = models.IntegerField(
        db_column="High2", blank=True, null=True
    )  # Field name made lowercase.
    high1 = models.IntegerField(
        db_column="High1", blank=True, null=True
    )  # Field name made lowercase.
    low1 = models.IntegerField(
        db_column="Low1", blank=True, null=True
    )  # Field name made lowercase.
    low2 = models.IntegerField(
        db_column="Low2", blank=True, null=True
    )  # Field name made lowercase.
    positivedialcreep = models.SmallIntegerField(
        db_column="PositiveDialCreep", blank=True, null=True
    )  # Field name made lowercase.
    negativedialcreep = models.SmallIntegerField(
        db_column="NegativeDialCreep", blank=True, null=True
    )  # Field name made lowercase.
    troublemessage = models.CharField(
        db_column="TroubleMessage", max_length=40, blank=True, null=True
    )  # Field name made lowercase.

    def delete(self):
        """
        Manage deletion manually as UniqueIdentifierField does not support on_delete="Cascade"
        """
        endpoints = Endpoint.objects.filter(meterPK=self.PK).all()
        if endpoints:
            for endpoint in endpoints:
                endpoint.delete()
        readings = Reading.objects.filter(meterPK=self.PK).all()
        if readings:
            for reading in readings:
                reading.delete()
        extendeddlrequests = Extendeddlrequest.objects.filter(meterPK=self.PK).all()
        if extendeddlrequests:
            for extendeddlrequest in extendeddlrequests:
                extendeddlrequest.delete()
        super(Meter, self).delete()
        return

    def __eq__(self, other):
        if self.__class__ == other.__class__:
            fields = [field.name for field in self._meta.fields]
            for field in fields:
                if not getattr(self, field) == getattr(other, field):
                    return False
            return True
        else:
            raise TypeError("Comparing object is not of the same type.")

    class Meta:
        managed = True
        db_table = "Meter"


class Meterevent(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    meterPK = models.ForeignKey(
        Meter, models.DO_NOTHING, db_column="MeterPK"
    )  # Field name made lowercase.
    type = models.CharField(db_column="Type", max_length=32)  # Field name made lowercase.
    timeoccurred = models.DateTimeField(db_column="TimeOccurred")  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "MeterEvent"


class Reading(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    # meterPK =   UniqueIdentifierField(primary_key=False, max_length=36, editable=False, default=None,  db_column='MeterPK')
    meter = models.ForeignKey(
        Meter, models.DO_NOTHING, db_column="MeterPK", related_name="reading"
    )  # Field name made lowercase.
    rawvalue = models.BigIntegerField(db_column="RawValue")  # Field name made lowercase.
    timeoccurred = models.DateTimeField(db_column="TimeOccurred")  # Field name made lowercase.
    ismanual = models.BooleanField(db_column="IsManual")  # Field name made lowercase.
    packettype = models.SmallIntegerField(db_column="PacketType")  # Field name made lowercase.
    packetdata = models.CharField(
        db_column="PacketData", max_length=100
    )  # Field name made lowercase.
    packettime = models.DateTimeField(db_column="PacketTime")  # Field name made lowercase.
    operatorusername = models.CharField(
        db_column="OperatorUserName", max_length=20
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "Reading"


class Receivercount(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    routestatisticpk = models.ForeignKey(
        "Routestatistic", models.DO_NOTHING, db_column="RouteStatisticPK"
    )  # Field name made lowercase.
    cardindex = models.SmallIntegerField(db_column="CardIndex")  # Field name made lowercase.
    value = models.IntegerField(db_column="Value")  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "ReceiverCount"
        unique_together = (("routestatisticpk", "cardindex"),)


class Route(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    identifier = models.CharField(
        db_column="Identifier", max_length=14
    )  # Field name made lowercase.
    primarystate = models.SmallIntegerField(db_column="PrimaryState")  # Field name made lowercase.
    secondarystate = models.SmallIntegerField(
        db_column="SecondaryState"
    )  # Field name made lowercase.
    unit = models.CharField(db_column="Unit", max_length=8)  # Field name made lowercase.
    company = models.CharField(db_column="Company", max_length=31)  # Field name made lowercase.
    frequencytext = models.CharField(
        db_column="FrequencyText", max_length=7
    )  # Field name made lowercase.
    operator = models.CharField(db_column="Operator", max_length=16)  # Field name made lowercase.
    tone = models.IntegerField(db_column="Tone")  # Field name made lowercase.
    frequency = models.FloatField(db_column="Frequency")  # Field name made lowercase.
    fromhostname = models.CharField(
        db_column="FromHostName", max_length=13
    )  # Field name made lowercase.
    originalertcount = models.IntegerField(
        db_column="OriginalErtCount"
    )  # Field name made lowercase.
    collectoutofroute = models.BooleanField(
        db_column="CollectOutOfRoute"
    )  # Field name made lowercase.
    definition = models.CharField(
        db_column="Definition", max_length=18
    )  # Field name made lowercase.
    sortflag = models.SmallIntegerField(db_column="SortFlag")  # Field name made lowercase.
    sorttype = models.SmallIntegerField(db_column="SortType")  # Field name made lowercase.
    checkdigit = models.SmallIntegerField(db_column="CheckDigit")  # Field name made lowercase.
    metertypesincluded = models.SmallIntegerField(
        db_column="MeterTypesIncluded"
    )  # Field name made lowercase.
    frequencyselector = models.SmallIntegerField(
        db_column="FrequencySelector"
    )  # Field name made lowercase.
    expectedrundate = models.DateTimeField(
        db_column="ExpectedRunDate"
    )  # Field name made lowercase.
    basefilename = models.CharField(
        db_column="BaseFileName", max_length=32
    )  # Field name made lowercase.
    fileformat = models.CharField(
        db_column="FileFormat", max_length=1
    )  # Field name made lowercase.
    ekaimportindicator = models.SmallIntegerField(
        db_column="EkaImportIndicator"
    )  # Field name made lowercase.
    maxmtrmsglength = models.SmallIntegerField(
        db_column="MaxMtrMsgLength"
    )  # Field name made lowercase.
    message = models.CharField(db_column="Message", max_length=160)  # Field name made lowercase.
    filespackaged = models.BooleanField(db_column="FilesPackaged")  # Field name made lowercase.
    dldatetimeformat = models.SmallIntegerField(
        db_column="DLDateTimeFormat"
    )  # Field name made lowercase.
    firstimportdate = models.DateTimeField(
        db_column="FirstImportDate"
    )  # Field name made lowercase.
    firststartprocessingdate = models.DateTimeField(
        db_column="FirstStartProcessingDate"
    )  # Field name made lowercase.
    firstinroutereaddate = models.DateTimeField(
        db_column="FirstInRouteReadDate"
    )  # Field name made lowercase.
    firstoorreaddate = models.DateTimeField(
        db_column="FirstOORReadDate"
    )  # Field name made lowercase.
    laststopprocessingdate = models.DateTimeField(
        db_column="LastStopProcessingDate"
    )  # Field name made lowercase.
    lastcompletedate = models.DateTimeField(
        db_column="LastCompleteDate"
    )  # Field name made lowercase.
    readingkeylistdate = models.DateTimeField(
        db_column="ReadingKeyListDate"
    )  # Field name made lowercase.
    readingkeylist = models.BinaryField(db_column="ReadingKeyList")  # Field name made lowercase.
    modechangetype = models.SmallIntegerField(
        db_column="ModeChangeType"
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "Route"


class Routemessage(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    routePK = models.ForeignKey(
        Route, models.DO_NOTHING, db_column="RoutePK"
    )  # Field name made lowercase.
    sequencenumber = models.IntegerField(db_column="SequenceNumber")  # Field name made lowercase.
    messageheader = models.CharField(
        db_column="MessageHeader", max_length=50
    )  # Field name made lowercase.
    message = models.CharField(db_column="Message", max_length=150)  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "RouteMessage"


class Routestatistic(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    routePK = models.ForeignKey(
        Route, models.DO_NOTHING, db_column="RoutePK"
    )  # Field name made lowercase.
    activetime = models.BigIntegerField(db_column="ActiveTime")  # Field name made lowercase.
    missedoutofroutereadcount = models.IntegerField(
        db_column="MissedOutOfRouteReadCount"
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "RouteStatistic"


class Routetoneoverride(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    routeidentifier = models.CharField(
        db_column="RouteIdentifier", max_length=14
    )  # Field name made lowercase.
    tone = models.IntegerField(db_column="Tone")  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "RouteToneOverride"


class Schemaversion(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    value = models.CharField(db_column="Value", max_length=20)  # Field name made lowercase.
    iscurrentversion = models.BooleanField(
        db_column="IsCurrentVersion"
    )  # Field name made lowercase.
    fpdatetime = models.DateTimeField(db_column="FpDateTime")  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "SchemaVersion"


class Segment(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    routePK = models.ForeignKey(
        Route, models.DO_NOTHING, db_column="RoutePK"
    )  # Field name made lowercase.
    identifier = models.IntegerField(db_column="Identifier")  # Field name made lowercase.
    definition = models.CharField(
        db_column="Definition", max_length=31
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "Segment"


class Settings(models.Model):
    PK = models.CharField(
        db_column="PK", primary_key=True, max_length=36
    )  # Field name made lowercase.
    readingkeylistdate = models.DateTimeField(
        db_column="ReadingKeyListDate"
    )  # Field name made lowercase.
    readingkeylist = models.BinaryField(db_column="ReadingKeyList")  # Field name made lowercase.
    readingkeypassphrase = models.BinaryField(
        db_column="ReadingKeyPassPhrase"
    )  # Field name made lowercase.

    class Meta:
        managed = True
        db_table = "Settings"

