
import os

from collections import namedtuple

os.environ.setdefault("DJANGO_SETTINGS_MODULE", "settings")
from django.test import TestCase

# from data.models import *
# from Analyzer import Analyzer
# from Updater import Updater
from cleargrid.mc3_integration.test.FakeData import FakeDataGenerator
from cleargrid.mc3_integration.poller import PollingThread, ORMFetcher
from queue import Queue


Row = namedtuple(
    "Row",
    ["identifier", "timeoccurred", "latitude", "longitude", "dl_requests"],
)


class MockORMFetcher:
    def __init__(self):
        self.last_realtime_fetch = None
        self.last_logging_fetch = None
        self.count = None
        self._meters = []
        self._realtime_messages = []
        self._logging_messages = []
        self._updated = 0

    @property
    def meters_updated(self) -> int:
        """Returns if a new meter exists"""
        return self._updated

    @property
    def realtime_messages(self):
        return self._realtime_messages

    @property
    def logging_messages(self):
        return self._logging_messages

    @property
    def meters(self):
        return self._meters

    def set_meter(self, value):
        self._meters = value


class TestPoller(TestCase):
    def setUp(self):
        """
        This is set up for every test
            1. Stub the data
            2. Initialize the Analyzer
            3. Create a random ENDPOINT, 
        """
        self.result_q = Queue()
        self.mock_orm_fetcher = MockORMFetcher()
        self.poller = PollingThread(
            None, result_q=self.result_q, fetcher=self.mock_orm_fetcher
        )

    def test_empty_poller(self):
        self.poller.process()
        self.assertEqual(self.result_q.empty(), True)

    def test_new_meter(self):
        # Run through an initial cycle
        self.poller.process()
        # Add meters and update
        self.mock_orm_fetcher.set_meter(
            [
                Row(
                    identifier="66052082",
                    latitude="51.208463",
                    longitude="-112.903899",
                    dl_requests=1,
                    timeoccurred=None,
                ),
                Row(
                    identifier="42181791",
                    latitude="51.290495",
                    longitude="-113.006506",
                    dl_requests=1,
                    timeoccurred=None,
                ),
            ]
        )
        self.mock_orm_fetcher._updated = 2
        self.poller.process()
        # Check the Queue
        command, data = self.result_q.get()
        map_command, map_data = self.result_q.get()
        required_output = {
            "66052082": {
                "identifier": "66052082",
                "latitude": "51.208463",
                "longitude": "-112.903899",
                "realtime": [],
                "logging": [],
            },
            "42181791": {
                "identifier": "42181791",
                "latitude": "51.290495",
                "longitude": "-113.006506",
                "realtime": [],
                "logging": [],
            },
        }
        map_required_output = {
            "66052082": {
                "identifier": "66052082",
                "latitude": "51.208463",
                "longitude": "-112.903899",
                "realtime": [],
                "logging": [],
            },
            "42181791": {
                "identifier": "42181791",
                "latitude": "51.290495",
                "longitude": "-113.006506",
                "realtime": [],
                "logging": [],
            },
        }
        self.assertEqual(command, "add_meters")
        self.assertEqual(data, required_output)
        self.assertEqual(map_command, "update_map")
        self.assertEqual(map_data, map_required_output)
    def test_new_scm(self):
        # Run through an initial cycle
        self.poller.process()
        # Add meters and update
        self.mock_orm_fetcher.set_meter(
            [
                Row(
                    identifier="66052082",
                    latitude="51.208463",
                    longitude="-112.903899",
                    dl_requests=1,
                    timeoccurred=None,
                ),
                Row(
                    identifier="42181791",
                    latitude="51.290495",
                    longitude="-113.006506",
                    dl_requests=1,
                    timeoccurred=None,
                ),
            ]
        )
        self.mock_orm_fetcher._updated = 2
        self.poller.process()
        # Clear the queue
        while not self.result_q.empty():
            self.result_q.get()

        # Add scm readings
        self.mock_orm_fetcher._realtime_messages = [
            Row(
                identifier="66052082",
                latitude="51.208463",
                longitude="-112.903899",
                dl_requests=1,
                timeoccurred=datetime(2019, 7, 2, 14, 46, 11, 760000),
            ),
            Row(
                identifier="42181791",
                latitude="51.290495",
                longitude="-113.006506",
                dl_requests=1,
                timeoccurred=datetime(2019, 7, 2, 15, 12, 8, 167000),
            ),
        ]
        self.poller.process()

        # Check the queue
        reading_command, reading_data = self.result_q.get()
        map_command, map_data = self.result_q.get()
        required_output = {
            "66052082": [("scm", 1562100371.76)],
            "42181791": [("scm", 1562101928.167)],
        }
        map_required_output = {
            "66052082": {
                "identifier": "66052082",
                "latitude": "51.208463",
                "longitude": "-112.903899",
                "realtime": [("scm", 1562100371.76)],
                "logging": [],
            },
            "42181791": {
                "identifier": "42181791",
                "latitude": "51.290495",
                "longitude": "-113.006506",
                "realtime": [("scm", 1562101928.167)],
                "logging": [],
            },
        }
        self.assertEqual(reading_command, "update_readings")
        self.assertEqual(reading_data, required_output)
        self.assertEqual(map_command, "update_map")
        self.assertEqual(map_data, map_required_output)

    def test_new_idm(self):
        # Run through an initial cycle
        self.poller.process()
        # Add meters and update
        self.mock_orm_fetcher.set_meter(
            [
                Row(
                    identifier="66052082",
                    latitude="51.208463",
                    longitude="-112.903899",
                    dl_requests=1,
                    timeoccurred=None,
                ),
                Row(
                    identifier="42181791",
                    latitude="51.290495",
                    longitude="-113.006506",
                    dl_requests=1,
                    timeoccurred=None,
                ),
            ]
        )
        self.mock_orm_fetcher._updated = 2
        self.poller.process()
        # Clear the queue
        while not self.result_q.empty():
            self.result_q.get()

        # Add scm readings
        self.mock_orm_fetcher._logging_messages = [
            Row(
                identifier="66052082",
                latitude="51.208463",
                longitude="-112.903899",
                dl_requests=1,
                timeoccurred=datetime(2019, 7, 2, 14, 46, 11, 760000),
            ),
            Row(
                identifier="42181791",
                latitude="51.290495",
                longitude="-113.006506",
                dl_requests=1,
                timeoccurred=datetime(2019, 7, 2, 15, 12, 8, 167000),
            ),
        ]
        self.poller.process()

        # Check the queue
        reading_command, reading_data = self.result_q.get()
        map_command, map_data = self.result_q.get()
        required_output = {
            "66052082": [("idm", 1562100371.76)],
            "42181791": [("idm", 1562101928.167)],
        }
        map_required_output = {
            "66052082": {
                "identifier": "66052082",
                "latitude": "51.208463",
                "longitude": "-112.903899",
                "realtime": [],
                "logging": [("idm", 1562100371.76)],
            },
            "42181791": {
                "identifier": "42181791",
                "latitude": "51.290495",
                "longitude": "-113.006506",
                "realtime": [],
                "logging": [("idm", 1562101928.167)],
            },
        }
        self.assertEqual(reading_command, "update_readings")
        self.assertEqual(reading_data, required_output)
        self.assertEqual(map_command, "update_map")
        self.assertEqual(map_data, map_required_output)


class TestORMFetcher(TestCase):
    def setUp(self):
        """
        This is set up for every test
            1. Stub the data
            2. Initialize the Analyzer
            3. Create a random ENDPOINT, 
        """
        self.orm_fetcher = ORMFetcher()
        self.fake_data_generator = FakeDataGenerator()

    def test_meters_updated(self):
        # Count initial database data
        self.orm_fetcher.meters_updated
        # Add a meter
        self.fake_data_generator.create_endpoint()
        # Meters Updated should be True
        self.assertTrue(self.orm_fetcher.meters_updated)
        # If we check again, it should be false
        self.assertFalse(self.orm_fetcher.meters_updated)

    def test_realtime_messages(self):
        # Get initial messages (empty list)
        messages = self.orm_fetcher.realtime_messages
        self.assertFalse(messages)

        # Add a message
        endpoint, meter = self.fake_data_generator.create_endpoint()
        generated_reading = self.fake_data_generator.create_reading()

        # Get updated messages
        messages = self.orm_fetcher.realtime_messages

        # Assertion
        self.assertEqual(
            messages[0].timeoccurred,
            self.orm_fetcher.last_realtime_fetch,
            generated_reading.timeoccurred,
        )
        self.assertEqual(str(meter.identifier), messages[0].identifier)

    def test_logging_messages(self):
        # Get initial messages (empty list)
        messages = self.orm_fetcher.logging_messages
        self.assertFalse(messages)

        # Add a message
        endpoint, meter = self.fake_data_generator.create_endpoint()
        generated_request = self.fake_data_generator.create_extendeddlRequest()

        # Get updated messages
        messages = self.orm_fetcher.logging_messages
        self.assertFalse(messages)

        # Generate Reading
        generated_dlreading = self.fake_data_generator.create_extendeddlReading(
            generated_request
        )

        # Get updated messages
        messages = self.orm_fetcher.logging_messages

        # Assertion
        self.assertEqual(
            messages[0].timeoccurred,
            self.orm_fetcher.last_logging_fetch,
            generated_dlreading.timeoccurred,
        )
        self.assertEqual(str(meter.identifier), messages[0].identifier)


# class TestAnalyzer(TestCase):
#     def setUp(self):
#         """
#         This is set up for every test
#             1. Stub the data
#             2. Initialize the Analyzer
#             3. Create a random ENDPOINT,
#         """
#         self.requests_mock = mock.patch("Updater.requests.post")
#         self.requests_patch = self.requests_mock.start()
#         self.analyzer = Analyzer({}, {}, {}, {}, {})
#         self.updater = Updater({})
#         self.data_generator = FakeDataGenerator(unit_test_override=True)

#     def tearDown(self):
#         self.requests_patch.stop

#     def test_analyzer_single(self):
#         """
#         1. Creating an Endpoint -> Show up as Endpoint on UI
#         2. Creating a Reading -> Remove Endpoint from UI
#         3. Create an ExtendedReading -> Remove Endpoint from UI
#         """
#         creation_endpoints = {}
#         removal_endpoints = {}

#         # 1. CREATE a meter/endpoint
#         created_endpoint = self.data_generator.create_data(override="endpoint")
#         endpoint_by_pk = {e.pk: e for e in Endpoint.objects.all()}
#         creation_endpoints["endpoints"], removal_endpoints["endpoints"] = self.analyzer.diff_check(
#             endpoint_by_pk, "ENDPOINTS"
#         )
#         # Analyzer prompts to add endpoint to UI
#         self.assertIn(str(created_endpoint.pk), {e.pk for e in creation_endpoints["endpoints"]})

#         # 2. CREATE a Reading associated with an Endpoint
#         # (An endpoint has received a reading)
#         reading_endpoint = self.data_generator.create_data(override="readings")
#         reading_by_pk = {e.pk: e for e in Reading.objects.all()}
#         creation_endpoints["readings"], removal_endpoints["readings"] = self.analyzer.diff_check(
#             reading_by_pk, "READINGS"
#         )
#         # Analyzer prompts to remove endpoint from UI
#         self.assertIn(str(reading_endpoint.pk), {e.pk for e in removal_endpoints["readings"]})

#         # 3. CREATE an extended_reading associated with an Endpoint
#         # (An endpoint has received an extended_reading)
#         extended_reading = self.data_generator.create_data(override="extended_readings")
#         extended_reading_by_pk = {e.pk: e for e in Extendedreading.objects.all()}
#         creation_endpoints["extended_readings"], removal_endpoints[
#             "extended_readings"
#         ] = self.analyzer.diff_check(extended_reading_by_pk, "EXTENDED_READINGS")
#         # Analyzer prompts to remove endpoint from UI
#         self.assertIn(
#             str(extended_reading.pk), {e.pk for e in removal_endpoints["extended_readings"]}
#         )

#     def test_analyzer_multiple(self):
#         """
#         1. Creating 50 Endpoints -> Show up as Endpoints on UI
#         2. Creating 50 Readings -> Remove Endpoints from UI
#         3. Create 50 ExtendedReadings -> Remove Endpoints from UI
#         """
#         creation_endpoints = {}
#         removal_endpoints = {}

#         # 1. CREATE 50 meter/endpoints
#         created_endpoints = []
#         for _ in range(50):
#             created_endpoints.append(self.data_generator.create_data(override="endpoint"))
#         endpoint_by_pk = {e.pk: e for e in Endpoint.objects.all()}
#         creation_endpoints["endpoints"], removal_endpoints["endpoints"] = self.analyzer.diff_check(
#             endpoint_by_pk, "ENDPOINTS"
#         )
#         # Analyzer prompts to add endpoint to UI
#         for created_endpoint in created_endpoints:
#             self.assertIn(str(created_endpoint.pk), {e.pk for e in creation_endpoints["endpoints"]})

#         # 2. CREATE 50 Readings associated with an Endpoint
#         # (50 endpoints have received a reading)
#         reading_endpoints = []
#         for _ in range(50):
#             reading_endpoints.append(self.data_generator.create_data(override="readings"))
#         reading_by_pk = {e.pk: e for e in Reading.objects.all()}
#         creation_endpoints["readings"], removal_endpoints["readings"] = self.analyzer.diff_check(
#             reading_by_pk, "READINGS"
#         )
#         # Analyzer prompts to remove endpoint from UI
#         for reading in reading_endpoints:
#             self.assertIn(str(reading.pk), {e.pk for e in removal_endpoints["readings"]})

#         # 3. CREATE 50 extended_readings associated with an Endpoint
#         # (50 endpoints have received an extended_reading)
#         extended_reading_endpoints = []
#         for _ in range(50):
#             extended_reading_endpoints.append(
#                 self.data_generator.create_data(override="extended_readings")
#             )
#         extended_readings_by_pk = {e.pk: e for e in Extendedreading.objects.all()}
#         creation_endpoints["extended_readings"], removal_endpoints[
#             "extended_readings"
#         ] = self.analyzer.diff_check(extended_readings_by_pk, "EXTENDED_READINGS")
#         # Analyzer prompts to remove endpoint from UI
#         for extended_reading_endpoint in extended_reading_endpoints:
#             if extended_reading_endpoint is not None:
#                 self.assertIn(
#                     str(extended_reading_endpoint.pk),
#                     {e.pk for e in removal_endpoints["extended_readings"]},
#                 )

#     def test_updater(self,):
#         endpoints_to_add = []
#         for _ in range(50):
#             endpoints_to_add.append((self.data_generator.create_data(override="endpoint"), None))

#         # Endpoints to remove will have keys selected from endpoints_to_add
#         endpoints_to_remove = []
#         for _ in range(50):
#             endpoints_to_remove.append(
#                 (self.data_generator.create_data(override="extended_readings"), "idm")
#             )

#         print(endpoints_to_add)
#         endpoints_to_add_keys = set(
#             [str(item[0].pk) for item in endpoints_to_add if item[0] is not None]
#         )
#         endpoints_to_remove_keys = set(
#             [str(item[0].pk) for item in endpoints_to_remove if item[0] is not None]
#         )

#         self.updater.send_updates((endpoints_to_add, endpoints_to_remove))

#         removal_counter = 0
#         addition_counter = 0

#         for _, value in self.requests_patch.call_args[1]["json"].items():
#             if value["read"] == False:
#                 addition_counter += 1
#             else:
#                 removal_counter += 1

#         assert removal_counter == len(endpoints_to_remove_keys)
#         assert addition_counter == len(endpoints_to_add_keys - endpoints_to_remove_keys)


if __name__ == "__main__":
    unittest.main()
