import threading
from queue import Queue, Empty

from .updater import Updater
from .poller import PollingThread
import logging

logger = logging.getLogger(__name__)


class Runner(threading.Thread):
    def __init__(self, *args, poller=None, updater=None, **kwargs):

        polling_interval = kwargs.pop("polling_interval", None)
        api_server = kwargs.pop("api_server", None)

        super().__init__(*args, **kwargs)

        self._exit = threading.Event()
        self.result_q = Queue()

        if poller is None:
            poller = PollingThread(polling_interval, result_q=self.result_q)
        logging.debug("Using poller %r", poller)
        self.poller = poller

        if updater is None:
            updater = Updater(api_server)
        logging.debug("Using updater %r", updater)
        self.updater = updater

    def run(self):
        self.poller.start()
        try:
            while not self._exit.is_set():
                try:
                    method, update = self.result_q.get(False)
                except Empty:
                    logger.debug("Nothing to process in queue")
                    self._exit.wait(1)
                    continue
                getattr(self.updater, method)(update)

        except KeyboardInterrupt:
            logger.warning("Ctrl-c received! Killing threads...")
        finally:
            self.poller.join()

    def stop(self):
        self._exit.set()

