import struct
from typing import List
from commlib.proto.message_pb2 import Message
from commlib.proto.ping_pb2 import PingMessage
from commlib.proto.mission_status_pb2 import (
    PrepareMissionMessage,
    CompleteMissionMessage,
    MissionStateRequest,
    MissionStateMessage,
    MissionStatusSyncRequest,
    MissionStatusSyncMessage,
    MissionStatusUpdate,
    MissionStatusUpdateMessage,
)
from commlib.proto.location_pb2 import (
    NoLocationMessage,
    LocationMessage,
    LocationSyncMessage,
    LocationSyncRequest,
    TickRange,
)
from datetime import datetime, timezone


def chunkify(l, n):
    """Yield successive n-sized chunks from l."""
    for i in range(0, len(l), n):
        yield l[i : i + n]


def pack_bytes_sync_message(data):
    v = None
    for index, item in enumerate(data):
        if v == None:
            v = (item & 0x3) << 6
        else:
            v = v | (item & 0x3) << (6 - 2 * index)
        # v = (((a & 0x3) << 6) | ((b & 0x3) << 4) | ((c & 0x3) << 2) | (d & 0x3)) & 0xFF
    # return v.to_bytes(2, byteorder="big")
    return v


def pack_bytes_status_update(item):
    """
    ARGS:
        item (dict): {"index", "value"}
    """
    v = (item["index"] & 0x3FFFFF) << 2
    v = v | (item["value"] & 0x3)
    # v = (((a & 0x3) << 6) | ((b & 0x3) << 4) | ((c & 0x3) << 2) | (d & 0x3)) & 0xFF
    # return v.to_bytes(2, byteorder="big")
    return v


def unpack_bytes_sync_message(start, end, packed_message):
    """
    ARGS: 
        start (int): Update INCUDES start index
        end (int): Update EXCLUDES end index (up to end index)
        packed_message (bytes): Device state updates

    RETURNS:
        unpacked_messages (List): A list of unpacked messages

    """
    unpacked_messages = []

    list_repr = list(packed_message)

    for package in list_repr:
        bin_statuses = format(package, "08b")
        split_statuses = list(chunkify(bin_statuses, 2))
        for number in split_statuses:
            unpacked_messages.append(int(number, 2))

    # There is a chance that there will be 0 padding resulting in
    # more "readings" than wanted. We throw out any extra's
    truncated_messages = unpacked_messages[0 : (end - start)]

    enum = ["unheard", "heard", "complete", "error"]

    truncated_messages = [enum[index] for index in truncated_messages]

    return truncated_messages


def get_unix_time():
    return int(datetime.now(timezone.utc).timestamp())


class PayloadConstructor:
    @staticmethod
    def augment_payload(payload, payload_type, payload_message, serialized=False):
        """
        ARGS: 
            payload (Payload): payload which the `payload_message` will be added to
            payload_type (str): payload_type
            payload_message (message): One of message types in Payload matching `payload_type`
            serialized (bool): Whether the message has to be deserialized
        """
        # The following method not work as per
        # https://groups.google.com/forum/#!topic/protobuf/9nzKMsuX6tA
        # Proto3 requires building queries from the "bottom up"
        # setattr(payload, payload_type, payload_message)

        if serialized == True:
            if payload_type == "ping_message":
                deserializer = PingMessage()
            elif payload_type == "mission_status_sync_message":
                deserializer = MissionStatusSyncMessage()
            elif payload_type == "mission_status_update_message":
                deserializer = MissionStatusUpdateMessage()
            elif payload_type == "mission_state_request":
                deserializer = MissionStateRequest()
            elif payload_type == "mission_status_sync_request":
                deserializer = MissionStatusSyncRequest()
            elif payload_type == "prepare_mission_message":
                deserializer = PrepareMissionMessage()
            elif payload_type == "complete_mission_message":
                deserializer = CompleteMissionMessage()
            elif payload_type == "mission_state_message":
                deserializer = MissionStateMessage()
            elif payload_type == "location_message":
                deserializer = LocationMessage()
            elif payload_type == "no_location_message":
                deserializer = NoLocationMessage()
            elif payload_type == "location_sync_request":
                deserializer = LocationSyncRequest()
            elif payload_type == "location_sync_message":
                deserializer = LocationSyncMessage()
            deserializer.ParseFromString(payload_message)
            payload_message = deserializer

        if payload_type == "ping_message":
            payload.ping.CopyFrom(payload_message)
        elif payload_type == "mission_status_sync_message":
            payload.mission_status_sync_message.CopyFrom(payload_message)
        elif payload_type == "mission_status_update_message":
            payload.mission_status_update_message.CopyFrom(payload_message)
        elif payload_type == "mission_state_request":
            payload.mission_state_request.CopyFrom(payload_message)
        elif payload_type == "mission_status_sync_request":
            payload.mission_status_sync_request.CopyFrom(payload_message)
        elif payload_type == "prepare_mission_message":
            payload.prepare_mission_message.CopyFrom(payload_message)
        elif payload_type == "complete_mission_message":
            payload.complete_mission_message.CopyFrom(payload_message)
        elif payload_type == "mission_state_message":
            payload.mission_state_message.CopyFrom(payload_message)
        elif payload_type == "location_message":
            payload.location_message.CopyFrom(payload_message)
        elif payload_type == "no_location_message":
            payload.no_location_message.CopyFrom(payload_message)
        elif payload_type == "location_sync_request":
            payload.location_sync_request.CopyFrom(payload_message)
        elif payload_type == "location_sync_message":
            payload.location_sync_message.CopyFrom(payload_message)
        return payload

    @staticmethod
    def create_ping(serialize=False):
        """Ping Me Up Scottie"""
        ping_message = PingMessage()
        ping_message.current_time = get_unix_time()

        if serialize == True:
            ping_message = ping_message.SerializeToString()

        return ping_message

    @staticmethod
    def create_mission_status_sync_message(
        mission_number: int,
        start_index: int,
        end_index: int,
        collection_status=None,
        serialize=False,
    ):
        """ 
        Args: 
            mission_number (int)
            start_index (int)
            end_index (int)
            collection_status (List): Only if is a RESPONSE
        """
        sync_message = MissionStatusSyncMessage()
        if mission_number is not None:
            sync_message.mission_number = mission_number
        sync_message.start = start_index
        sync_message.end = end_index
        if collection_status is not None:
            # This is a response
            sync_message.status.extend(collection_status)
            chunked_data = chunkify(collection_status, 4)
            packed_data = [pack_bytes_sync_message(item) for item in chunked_data]
            sync_message.packed_status = bytes(packed_data)

        if serialize == True:
            sync_message = sync_message.SerializeToString()

        return sync_message

    @staticmethod
    def create_mission_status_update_message(
        mission_number: int, status_updates: list = None, serialize=False
    ):
        """
        Args: 
            mission_number (int)
            status_updates (List)
        """
        status_update = MissionStatusUpdateMessage()
        statuses = []
        status_update.mission_number = mission_number
        for index, item in enumerate(status_updates):
            new_status = MissionStatusUpdate()
            new_status.index = item["index"]
            new_status.status = item["value"]
            statuses.append(new_status)

        # Manage Packed Status
        packed_data = bytes([pack_bytes_status_update(item) for item in status_updates])
        status_update.packed_status = packed_data
        status_update.status.extend(statuses)

        if serialize == True:
            status_update = status_update.SerializeToString()

        return status_update

    @staticmethod
    def create_mission_state_request(mission_number: int, serialize=False):
        """
        Request the collector send the mission_state
        """
        sync_request = MissionStateRequest()
        if mission_number is not None:
            sync_request.mission_number = mission_number

        if serialize == True:
            sync_request = sync_request.SerializeToString()

        return sync_request

    @staticmethod
    def create_mission_status_sync_request(
        mission_number: int, start: int, end: int, serialize=False
    ):
        """
        Request that the collector service send the full status of the mission
        """
        sync_request = MissionStatusSyncRequest()
        if mission_number is not None:
            sync_request.mission_number = mission_number
        if start is not None and end is not None:
            sync_request.start = start
            sync_request.end = end

        if serialize == True:
            sync_request = sync_request.SerializeToString()

        return sync_request

    @staticmethod
    def create_prepare_mission_message(
        mission_number: int, mission_data_url: str, serialize=False
    ):
        """
        Request the collector service prepare the collector for the specified mission.
        """
        prepare_mission_message = PrepareMissionMessage()
        prepare_mission_message.mission_number = mission_number
        prepare_mission_message.mission_data_url = mission_data_url

        if serialize == True:
            prepare_mission_message = prepare_mission_message.SerializeToString()

        return prepare_mission_message

    @staticmethod
    def create_complete_mission_message(
        mission_number: int, mission_data_url: str, serialize=False
    ):
        """
        Request that the collector service stop collecting the mission, collect the 
        results together, and send the results to the LOB
        """
        complete_mission_message = CompleteMissionMessage()
        complete_mission_message.mission_number = mission_number
        complete_mission_message.mission_data_url = mission_data_url

        if serialize == True:
            complete_mission_message = complete_mission_message.SerializeToString()

        return complete_mission_message

    @staticmethod
    def create_mission_state_message(
        mission_number: int, mission_state, msg: str = None, serialize=False
    ):
        """
        Sent by the collector when it finishes each stage of the mission lifecycle. 
        May also be sent if collector wants to update msg string displayed while 
        in the samemstate.
        """
        mission_state_message = MissionStateMessage()
        if mission_number is not None:
            mission_state_message.mission_number = mission_number
        mission_state_message.state = mission_state
        if msg is not None:
            mission_state_message.msg = msg

        if serialize == True:
            mission_state_message = mission_state_message.SerializeToString()

        return mission_state_message

    @staticmethod
    def create_location_message(
        latitude: float,
        longitude: float,
        altitude: int,
        heading: int,
        ground_speed: int,
        tick: int,
        serialize=False,
    ):
        """
        Regular message containing the current location of the collector. 
        Max of 1 per second.

        All fields will be set to None when creating a `location_sync_message` and the `tick`
        returns an invalid or no_location
        """
        location_message = LocationMessage()
        location_message.current_time = tick
        if latitude is not None:
            location_message.latitude = int(latitude * (10 ** 7))
        if longitude is not None:
            location_message.longitude = int(longitude * (10 ** 7))
        if altitude is not None:
            location_message.altitude = int(altitude)
        if heading is not None:
            location_message.heading = int(heading)
        if ground_speed is not None:
            location_message.ground_speed = int(ground_speed)

        if serialize == True:
            location_message = location_message.SerializeToString()

        return location_message

    @staticmethod
    def create_no_location_message(
        sv_data,
        last_position: LocationMessage,
        antenna_connected: bool,
        serialize=False,
    ):
        no_location_message = NoLocationMessage()
        no_location_message.current_time = get_unix_time()
        no_location_message.sv_data = bytes(sv_data, "utf-8")
        no_location_message.antenna_connected = antenna_connected

        if serialize == True:
            no_location_message = no_location_message.SerializeToString()

        return no_location_message

    @staticmethod
    def create_location_sync_request(tick_ranges: List[List[int]], serialize=False):
        """
        Regular message containing the current location of the collector. 
        Max of 1 per second.
        ARGS: 
            tick_ranges (List): [[start_tick, end_tick],[start_tick, end_tick]]   Which ticks are to be searched for INCLUSIVE
            start_tick (int)
            end_tick (int)
        """
        location_sync_request = LocationSyncRequest()
        for tick_range in tick_ranges:
            tick_range_message = TickRange()
            tick_range_message.start = tick_range[0]
            tick_range_message.end = tick_range[1]
            location_sync_request.tick_ranges.extend([tick_range_message])

        if serialize == True:
            location_sync_request = location_sync_request.SerializeToString()

        return location_sync_request

    @staticmethod
    def create_location_sync_message(
        location_messages: List[LocationMessage],
        no_locations: List[NoLocationMessage],
        serialize=False,
    ):
        """
        Regular message containing the current location of the collector. 
        Max of 1 per second.
        """
        location_sync_message = LocationSyncMessage()
        for location_message in location_messages:
            location_sync_message.locations.extend(location_message)
        for no_location in no_locations:
            location_sync_message.no_locatons.extend(no_location)

        if serialize == True:
            location_sync_message = location_sync_message.SerializeToString()

        return location_sync_message
