"""Basic class that gives us a uniform api for subprocesses"""

import logging
import select
import subprocess
import threading
import types
from subprocess import DEVNULL, PIPE


def polled_pipe_logger(self):
    # https://stackoverflow.com/a/49500729
    lines = b""
    poller = select.poll()
    poller.register(self.pipe, select.POLLHUP | select.POLLIN)
    while True:
        for descriptor, mask in poller.poll(-1):
            if descriptor == self.pipe.fileno():
                if mask & select.POLLIN:
                    r = self.pipe.read()
                    lines += r
                    while b"\n" in lines:
                        line, lines = lines.split(b"\n", 1)
                        self.logger.info("%s", line.decode())
                elif mask & select.POLLHUP:
                    self.logger.debug("Hang up")
                    return
                else:
                    self.logger.error("Unknown mask: 0x%x", mask)
            else:
                self.logger.error("Unknown descriptor: 0x%x", descriptor)


def default_pipe_logger(self):
    # https://stackoverflow.com/a/49500729
    lines = b""
    while True:
        r = self.pipe.read(32)
        lines += r
        while b"\n" in lines:
            line, lines = lines.split(b"\n", 1)
            self.logger.info("%s", line.decode())
        if not r:
            self.logger.info("%s", lines.decode())
            self.logger.error("PIPE CLOSED -- Assuming process has terminated")
            return


def pipe_logger_factory():
    if hasattr(select, "poll"):
        logging.info("Using Polled pipe logger")
        return polled_pipe_logger
    else:
        logging.info("Using default pipe logger")
        return default_pipe_logger


class PipeFollower(threading.Thread):
    def __init__(self, name, pname, pipe, pipe_logger=None):
        threading.Thread.__init__(
            self, name=f"PipeFollower-{name}-{pname}", daemon=True
        )
        self.pipe = pipe
        self.logger = logging.getLogger(
            ".".join([self.__class__.__name__, name, pname])
        )
        if pipe_logger is None:
            pipe_logger = pipe_logger_factory()
        self.pipe_logger = types.MethodType(pipe_logger, self)

    def run(self):
        try:
            if hasattr(self.pipe, "read"):
                return self.pipe_logger()
            else:
                with open(self.pipe, "rt", buffering=1) as read:
                    self.pipe = read
                    return self.run()
        except Exception:
            self.logger.exception("Unhandled Exception")
            raise


class ProcessManager(object):
    def __init__(self, name="", args=[]):
        if name:
            self.name = name
        else:
            self.name = self.__class__.__name__
        self.args = args
        self.logger = logging.getLogger(
            ".".join([self.__class__.__name__, name])
        )
        self.process = None
        self._stdout_follower = None
        self._stderr_follower = None

    def start(self):
        self.logger.info("Spawning: %r", self.args)
        self.process = subprocess.Popen(
            self.args, stdin=DEVNULL, stdout=PIPE, stderr=PIPE
        )
        self._stdout_follower = PipeFollower(
            self.name, "stdout", self.process.stdout
        )
        self._stdout_follower.start()
        self._stderr_follower = PipeFollower(
            self.name, "stderr", self.process.stderr
        )
        self._stderr_follower.start()
        return self

    def reload(self):
        ...

    def shutdown(self):
        if self.process:
            self.process.terminate()
            self.process.wait()

    def __str__(self):
        return "{name}({args!r}) -> pid {pid}".format(
            name=self.__class__.__name__, args=self.args, pid=self.process.pid
        )

    def is_alive(self):
        self.process.poll()
        # A None value indicates that the process hasn’t terminated yet
        return self.process.returncode is None
