import atexit
import json
import logging
import glob
import os
import os.path
import uuid
from collections import namedtuple
from datetime import datetime

from .config import config

Location = namedtuple("Location", "x y")


class Meter(object):
    def __init__(self, ert, scm=True, idm=False, location=None):
        self.ert = ert
        self.location = location
        self.scm = [] if scm else None
        self.idm = [] if idm else None

    @classmethod
    def from_dict(cls, d):
        try:
            ert = d["ert"]
            location = Location(d["location"][1], d["location"][0])
            scm = d.get("scm", True)
            idm = d.get("idm", False)
        except KeyError as exc:
            logging.error(exc)
            raise ValueError()
        return cls(ert, scm, idm, location)

    def __repr__(self):
        return "%s(%r, scm=%r, idm=%r, location=%r)" % (
            self.__class__.__name__,
            self.ert,
            self.scm is not None,
            self.idm is not None,
            self.location,
        )

    def add_reading(self, reading_type, timestamp):
        readings = getattr(self, reading_type, None)
        if readings is None:
            return
        readings.append(timestamp)

    @property
    def scm_needed(self):
        return self.scm is not None and self.scm

    @property
    def scm_read(self):
        return self.scm is None or self.scm

    @property
    def scm_requested(self):
        return self.scm is not None

    @property
    def idm_needed(self):
        return self.idm is not None and self.idm

    @property
    def idm_read(self):
        return self.idm is None or self.idm

    @property
    def idm_requested(self):
        return self.idm is not None

    @property
    def complete(self):
        return self.scm_read and self.idm_read


class Trackpoint(object):
    def __init__(self, location, timestamp):
        self.location = location
        self.timestamp = timestamp

    def to_dict(self):
        return {"location": self.location, "timestamp": self.timestamp}


class Bounds(object):
    def __init__(
        self,
        top_left=None,
        bottom_right=None,
        left=None,
        right=None,
        top=None,
        bottom=None,
    ):
        self.left = left
        self.right = right
        self.top = top
        self.bottom = bottom
        if top_left:
            self.top_left = top_left
        if bottom_right:
            self.bottom_right = bottom_right

    @property
    def top_left(self):
        return Location(self.left, self.top)

    @top_left.setter
    def top_left(self, top_left):
        self.top = top_left.y
        self.left = top_left.x

    @property
    def bottom_right(self):
        return Location(self.right, self.bottom)

    def __contains__(self, other):
        return (self.right >= other[0] >= self.left) and (
            self.top >= other[1] >= self.bottom
        )

    def __bool__(self):
        return (
            self.top is not None
            and self.bottom is not None
            and self.left is not None
            and self.right is not None
        )

    @bottom_right.setter
    def bottom_right(self, bottom_right):
        self.bottom = bottom_right.y
        self.right = bottom_right.x

    def __getitem__(self, key):
        if key == 0:
            return self.top_left
        if key == 1:
            return self.bottom_right
        raise IndexError()

    def __iter__(self):
        yield self.top_left
        yield self.bottom_right

    def __len__(self):
        return 2

    def _update_location(self, location):
        if self.left is None or location.x < self.left:
            self.left = location.x
        if self.top is None or location.y > self.top:
            self.top = location.y
        if self.right is None or location.x > self.right:
            self.right = location.x
        if self.bottom is None or location.y < self.bottom:
            self.bottom = location.y

    def _update_bounds(self, bounds):
        if self.left is None or bounds.left < self.left:
            self.left = bounds.left
        if self.top is None or bounds.top > self.top:
            self.top = bounds.top
        if self.right is None or bounds.right > self.right:
            self.right = bounds.right
        if self.bottom is None or bounds.bottom < self.bottom:
            self.bottom = bounds.bottom

    def update(self, other):
        if isinstance(other, Location):
            self._update_location(other)
        elif isinstance(other, Bounds) and other:
            self._update_bounds(other)

    def __repr__(self):
        return "Bounds(top=%r, left=%r, bottom=%r, right=%r)" % (
            self.top,
            self.left,
            self.bottom,
            self.right,
        )


# TODO: add timestamp of most recent update to the data
# TODO: look at the adding of a lock to the dataset to prevent race conditions.


class DatastoreClass(object):
    def __init__(self):
        self.track = []
        self.previous_tracks = []
        self.meters = {}
        self.meter_bounds = Bounds()
        self.track_bounds = Bounds()
        self.kv = {}
        self.flight_id = str(uuid.uuid4())[:8]
        self.datastore_path = config.datastore_path
        self.clean_files()

    def completion(self, read_type):
        meters = [
            m
            for m in Datastore.meters.values()
            if getattr(m, read_type) is not None
        ]
        count = len(meters)
        complete = len([1 for m in meters if getattr(m, read_type)])
        pct = 100
        if count:
            pct = (complete / count) * 100
        status = "fail"
        if pct > 90:
            status = "warn"
        if pct > 98:
            status = "good"
        return {
            "message": "%s :%d/%d (%0.5f%%)"
            % (read_type.upper(), complete, count, pct),
            "class": status,
        }

    @property
    def bounds(self):
        bounds = Bounds()
        bounds.update(self.meter_bounds)
        bounds.update(self.track_bounds)
        return bounds

    @property
    def location(self):
        if self.track:
            return self.track[-1]
        return None

    @location.setter
    def location(self, value):
        if isinstance(value, Trackpoint):
            self.track.append(value)
            self.track_bounds.update(value.location)
        else:
            raise ValueError(
                "Location must be an instance ot Trackpoint not %r" % value
            )

    def add_meter(self, meter):
        self.meters[meter.ert] = meter
        self.meter_bounds.update(meter.location)

    def remove_meter(self, meter_id):
        del self.meters[meter_id]

    def reset(self):
        self.track = []
        self.previous_tracks = []
        self.meters = {}
        self.meter_bounds = Bounds()
        self.track_bounds = Bounds()
        self.kv = {}

    def add_reading(self, ert, reading_type, timestamp):
        try:
            self.meters[ert].add_reading(reading_type, timestamp)
        except KeyError:
            logging.warning("ERT %r does not exist", ert)

    def add_key(self, key, value):
        self.kv[key] = value

    def keys(self):
        return self.kv.keys()

    def get_key(self, key):
        return self.kv.get(key, None)

    def get_archive_files(self):
        def parse_filename(path):
            filename = os.path.basename(path)
            filename, _ = os.path.splitext(filename)
            track_id, timestamp = filename.split("-", 1)
            timestamp = datetime.fromisoformat(timestamp.replace("_", ":"))
            return path, track_id, timestamp

        return {
            track_id: {
                "path": path,
                "track_id": track_id,
                "timestamp": timestamp,
            }
            for path, track_id, timestamp in (
                parse_filename(path)
                for path in glob.glob(
                    os.path.join(self.datastore_path, "*.json")
                )
            )
        }

    def persist_tracks_to_disk(self):
        """
        Writes a file name with the structure `flight_id-YYYY-MM-DD`
        It first checks to see if a file starting with `flight_id` exists
        """
        if not os.path.exists(self.datastore_path):
            os.makedirs(self.datastore_path)

        track = self.generate_json()
        if not track:
            return

        try:
            file_path = self.get_archive_files()[self.flight_id]["path"]
        except (KeyError):
            timestamp = datetime.now().isoformat().replace(":", "_")
            file_path = os.path.join(
                self.datastore_path, f"{self.flight_id}-{timestamp}.json"
            )

        with open(file_path, "w") as f:
            f.write(json.dumps(track, indent=2))

    def clean_files(self):
        """
        Removes any files that are greater than 5 days old
        """
        for archive in self.get_archive_files().values():
            if (datetime.today() - archive["timestamp"]).days > 5:
                os.remove(archive["path"])

    def generate_json(self, label=""):
        if len(self.track) > 0:
            return {
                "type": "LineString",
                "properties": {
                    "id": self.flight_id,
                    "start_time": self.track[0].timestamp,
                    "label": label,
                },
                "coordinates": [
                    (p.location.x, p.location.y) for p in self.track
                ],
            }
        return None


Datastore = DatastoreClass()


def exit_handler():
    """
    Exit handler saves all of the data in the Memorystore to a JSON file
    """
    if Datastore:
        Datastore.persist_tracks_to_disk()


atexit.register(exit_handler)
