#!/usr/bin/env python
"""Connect to a GPSD server and write the position information to stdout"""

import argparse
import logging
import signal
from datetime import datetime
from sys import stdout
from threading import Event

import argparsing

from .client import GPSDClient


def parser():
    parser = argparse.ArgumentParser()
    parser.add_argument('client', nargs='?', default='192.168.168.1')
    parser.add_argument('--stdout')
    parser.add_argument('--stderr')
    argparsing.install_loglevel(parser)

    return parser


def main(args):
    logging.basicConfig(
        format="%(asctime)s-%(filename)16s:%(lineno)4d-%(threadName)s{%(process)d}[%(levelname)4.4s]:%(message)s",
        level=args.log_level,
        filename=args.stderr,
    )
    client = GPSDClient(args.client)
    done = Event()

    signal.signal(signal.SIGTERM, lambda a,b: done.set())
    signal.signal(signal.SIGINT, lambda a,b: done.set())
    signal.signal(signal.SIGHUP, lambda a,b: done.set())
    # 2018-07-25 22:04:48.992 UTC
    if args.stdout:
        out = open(args.stdout, 'wb', buffering=0)
    else:
        out = stdout
    try:
        with client.watch(mode=3) as watch:
            for p in watch.stream:
                if p:
                    data = p.json
                else:
                    data = repr(p)
                out.write("{} {}\n".format(
                    datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S.%f %Z'),
                    data).encode('UTF-8')
                )
                if done.is_set():
                    return 0
    finally:
        if args.stdout:
            out.close()


if __name__ == '__main__':
    exit(main(parser().parse_args()))
