import datetime
import logging
import typing as _t
from collections import defaultdict
from dataclasses import dataclass, field

import arrow

_og = logging.getLogger(__name__)


@dataclass
class Task:
    task_id: str
    rest: dict

    def __hash__(self):
        return self.task_id.__hash__()

    @classmethod
    def from_json(cls, task):
        if "task_id" in task:
            return cls(task["task_id"], task)
        elif "headers" in task:
            return cls(task["headers"]["id"], task)
        raise NotImplementedError(f"{task!r}")


def _counter_dict():
    return defaultdict(int)


@dataclass
class Queue:
    name: str
    task_ids: _t.Set[str] = field(default_factory=set)
    task_names: _t.Dict[str, int] = field(default_factory=_counter_dict)
    task_names_deduplicated: _t.Dict[str, int] = field(
        default_factory=_counter_dict
    )
    oldest: arrow.Arrow = field(default_factory=arrow.get)

    def __hash__(self):
        return self.name.__hash__()

    def add(self, h):
        _og.debug("Recording %s(%s)", h["task"], h["id"])
        if h["id"] not in self.task_ids:
            self.task_names_deduplicated[h["task"]] += 1
        self.task_names[h["task"]] += 1
        self.task_ids.add(h["id"])
        eta = arrow.get(h["eta"])
        if eta < self.oldest:
            self.oldest = eta


@dataclass
class State:
    tasks: _t.Set[Task] = field(default_factory=set)
    queues: _t.Set[Queue] = field(default_factory=set)

    def __repr__(self):
        return "{name}(tasks={{{tasks}}}, queues={{{queues}}})".format(
            name=self.__class__.__name__,
            tasks=self._repr_tasks(),
            queues=self._repr_queues(),
        )

    def _repr_tasks(self):
        ret = ",\n\t".join((repr(task) for task in self.tasks))
        if ret:
            return "\n\t" + ret
        else:
            return ret

    def _repr_queues(self):
        ret = ",\n\t".join((repr(queue) for queue in self.queues))
        if ret:
            return "\n\t" + ret
        else:
            return ret
