import importlib.util
import logging
import pkgutil
from itertools import chain

logger = logging.getLogger(__name__)

CLIENTS = {}

for finder, name, ispkg in pkgutil.iter_modules(__path__):
    if not ispkg:
        continue
    try:
        spec = finder.find_spec(f"{__name__}.{name}")
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        CLIENTS[name] = module.Clients
    except AttributeError:
        pass
    except Exception:
        logger.exception("Unable to load %r", name)

logger.info("Loaded clients %r", CLIENTS)


def client_lookup(name):
    return CLIENTS.get(name)


def get_client(host=None, port=None, clients=None, **kwargs):
    if not clients:
        clients = CLIENTS.keys()

    clients = [
        client
        for client in chain.from_iterable(client_lookup(c) for c in clients)
        if client
    ]

    for Client in clients:
        try:
            logging.info("Attempting to connect using %r", Client)
            return Client(host=host, port=port, **kwargs)
        except Exception as exc:
            logger.warning("Unable to instantiate %r", Client, exc_info=exc)
    return None
