
from logging import StreamHandler
from typing import TYPE_CHECKING

from cleargrid.signal_masking import mask_signals

if TYPE_CHECKING:

    class _SafeEmitterBase:
        def __init__(self, name):
            # 'name' is a kwarg on the base handlers so we don't need
            # **kwargs preventing mypy from detecting bad keywords
            pass

        def emit(self, record):
            pass


else:
    _SafeEmitterBase = object


class SignalSafeEmitter(_SafeEmitterBase):
    """Delay select signals until after logging emit

    We use signals to control processes, with one of these signals raising
    an exception that should interrupt the flow of the code.  If the logger
    eats the exception (for its own very good reason) the code never knows
    that it got the signal to exit and then carries on as if nothing
    happened."""

    def __init__(self, *, signals, **kwargs):
        self.signals = signals
        super().__init__(**kwargs)

    def emit(self, record):
        with mask_signals(self.signals):
            return super().emit(record)


class SignalSafeStreamHandler(SignalSafeEmitter, StreamHandler):
    """Delay selected signals until after logging emit is completed"""
