import logging
import os
import time
import platform
from pathlib import Path

import appdirs
import configargparse
from cleargrid.logging.userapp import add_log_file
from cleargrid.util.subprocess import ProcessManager

FORMAT = (
    "%(asctime)s - %(filename)12s:%(lineno)4d - "
    "%(threadName)10s[%(levelname)7s]:%(message)s"
)


def dead(process):
    if not process.is_alive():
        logging.error("%s process unexpectedly exited", process.name)
        return True
    return False


def logs():
    # open logs directory from appdirs
    log_path = (
        Path(appdirs.user_log_dir("abu collector", "cleargrid"))
        .absolute()
        .parent.parent
    )
    print(log_path)
    if platform.system == "Windows":
        ProcessManager(
            "Open Logs Directory",
            [os.environ["COMSPEC"], "/C", f"start {log_path}"],
        ).start()


def start():
    p = configargparse.get_arg_parser(
        default_config_files=["/etc/cleargrid", "~/.cleargrid"]
    )
    p.add("-c", "--config", is_config_file=True, help="config file path")
    p.add_argument("--log-level", "-L", default="DEBUG", env_var="LOG_LEVEL")
    p.add_argument("--log-file", type=str, default=True, env_var="LOG_FILE")
    p.add_argument("--logs", action="store_true")
    args = p.parse_known_args()[0]

    if args.logs:
        logs()
        return

    logging.basicConfig(level=getattr(logging, args.log_level.upper()))
    add_log_file(args.log_file, "abu collector")

    os.environ.setdefault("LOG_LEVEL", args.log_level.upper())
    os.environ.setdefault("LOG_FILE", str(args.log_file))

    logging.debug("starting processes")

    processes = (
        ProcessManager("Interface Server", ["mission_controller"]),
        ProcessManager("GPS Client", ["gps_client"]),
        ProcessManager("MC3 Integration", ["mc3_integration"]),
    )

    for p in processes:
        p.start()
        time.sleep(2)

    if platform.system == "Windows":
    ProcessManager(
        "Start Chrome",
            [
                os.environ["COMSPEC"],
                "/C",
                "start chrome http://127.0.0.1/app/",
            ],
    ).start()

    try:
        while True:
            time.sleep(5)
            if any(dead(p) for p in processes):
                break
    except Exception:
        logging.exception("Unhandled Exception")
    finally:
        while any(p.is_alive() for p in processes):
            for p in processes:
                p.shutdown()

    logging.info("All processes Exited")
