import logging
import logging.config
import typing as _t
import signal


def parse_log_level(level):
    if level == "NEVER":
        return 100
    else:
        return getattr(logging, level)


help_handlers = (
    "Where to send the log messages? "
    "Send as text to stderr ('console'). "
    "Send as structured json records to stderr ('structuredjson')."
)


def parse_log_handlers(handler):
    if handler == "stackdriver":
        return "structuredjson"
    return handler


def configure_logging(
    masked: _t.Set[signal.Signals], level: str, handler: str
):
    config = {
        "version": 1,
        "disable_existing_loggers": False,
        "formatters": {
            "verbose": {
                "format": "{asctime:s} - {filename:>18s}:{lineno:<4d}- "
                "{process:5d}/{threadName:10s} - "
                "{task_name}@{task_id} "
                "[{name}:{levelname:6.6s}] "
                "{message}",
                "style": "{",
                "class": (
                    "cleargrid.logging.formatters."
                    "celerymeta.CeleryMetaFormatter"
                ),
            },
            "gke": {
                "format": "{message}",
                "style": "{",
                "class": (
                    "cleargrid.logging.formatters."
                    "celerymeta.CeleryMetaFormatter"
                ),
            },
        },
        "handlers": {
            "stackdriver": {
                "level": "DEBUG",
                "class": (
                    "cleargrid.logging.handlers."
                    "structuredjson.StructuredJSONHandler"
                ),
                "formatter": "gke",
                "name": "stackdriver",
                "additional_fields": [
                    "filename",
                    "lineno",
                    "name",  # The logger's "name"
                    "process",
                    "task_id",
                    "task_name",
                    "threadName",
                ],
                "signals": masked,
            },
            "console": {
                "level": "DEBUG",
                "class": (
                    "cleargrid.logging.handlers."
                    "signalsafe.SignalSafeStreamHandler"
                ),
                "formatter": "verbose",
                "signals": masked,
            },
        },
        "loggers": {"fly": {"level": "INFO"}},
        "root": {"handlers": [handler], "level": level},
    }
    logging.config.dictConfig(config)
