import logging
import os
import django
import argparse

os.environ.setdefault("DJANGO_SETTINGS_MODULE", "cleargrid.mc3_integration.settings")
django.setup()

from cleargrid.mc3_integration.runner import Runner
from cleargrid.mc3_integration.updater import Updater


class StdoutUpdater(Updater):
    def post(self, *args, **kwargs):
        print(f"POST: {args[0]}")
        for k, v in kwargs.items():
            print(f"\t{k:10s}:{v}")


class NullUpdater(Updater):
    def post(self, *args, **kwargs):
        pass


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--stdout",
        action="store_true",
        help="Print updates to STDOUT rather than posting to server",
    )
    parser.add_argument("--null", action="store_true", help="suppress updater output entirely")
    parser.add_argument(
        "-L",
        "--log-level",
        dest="log_level",
        type=str,
        default="INFO",
        help="Log level to set logger to",
    )
    args = parser.parse_args()

    logging.basicConfig(
        format="%(asctime)s - %(filename)12s:%(lineno)4d - %(threadName)10s[%(levelname)7s]:%(message)s",
        level=getattr(logging, args.log_level.upper(), "INFO"),
    )

    p = {}
    if args.stdout:
        p.update(updater=StdoutUpdater())
    if args.null:
        p.update(updater=NullUpdater())

    runner = Runner(**p)
    try:
        runner.start()
        runner.join()
    except KeyboardInterrupt:
        logging.info("Received CTRL-C, clean shutdown")
    finally:
        runner.stop()
        runner.join()


if __name__ == "__main__":
    main()
