import json
import logging

from flask import (
    Flask,
    make_response,
    redirect,
    request,
)
from flask_cors import CORS
import pathlib

# from config import config
from .data import Datastore, Location, Meter, Trackpoint

app = Flask(
    __name__,
    static_folder=str(pathlib.Path(__file__).parent.absolute() / "app"),
)
CORS(app)


@app.route("/")
def root():
    """
    Redirect requests to root to the display
    """
    return redirect("/app/", code=302)


@app.route("/app/")
def app_index():
    """
    By default serve the index on /app/
    """
    return app.send_static_file("index.html")


@app.route("/app/<filepath>")
def app_root(filepath):
    """
    Serve files inside the /app/ url path
    """
    return app.send_static_file(filepath)


@app.route("/reset", methods=["POST"])
def reset():
    """
    reset the datastore to its empty state
    """
    Datastore.reset()
    return json.dumps(True)


@app.route("/remove", methods=["POST"])
def remove():
    """
    remove a list of meters from the datastore
    """
    logging.debug(request)
    data = request.get_json(force=True)
    logging.debug(data)
    for meter in data:
        Datastore.remove_meter(meter)
    return json.dumps(True)


@app.route("/update", methods=["POST"])
def update():
    """
    Location and reading updates get posted to this as a json object.

    Currently two keys are recognized
    location : A list of, in order, the timestamp, latitude and longitude
    readings : A dictionary of lists of readings, keyed on the ERT number.
               Each reading is a 2 element list containing the teading type
               and the timestamp at this point the  data store only
               understands 'SCM' and 'IDM' reading types
    """
    logging.debug(request)
    data = request.get_json(force=True)
    logging.debug(data)
    location = data.pop("location", None)

    if location:
        timestamp, y, x = location
        Datastore.location = Trackpoint(Location(x, y), timestamp)
    for ert, readings in data.pop("readings", {}).items():
        for reading in readings:
            Datastore.add_reading(ert, *reading)
    for ert in data.pop("meters", []):
        Datastore.add_meter(Meter.from_dict(ert))
    for key, value in data.items():
        Datastore.add_key(key, value)

    return json.dumps(True)


@app.route("/previous", methods=["GET"])
def previous():
    """
    Returns a list of previus flights by `flight_id-YYYY-MM-DD`
    """
    tracks = []

    tracks = [
        t
        for t in Datastore.get_archive_files().values()
        if t["track_id"] != Datastore.flight_id
    ]
    tracks.sort(key=lambda x: x["timestamp"], reverse=True)
    for t in tracks:
        t["timestamp"] = t["timestamp"].isoformat()

    return json.dumps(tracks)


@app.route("/previous/<track_id>", methods=["GET"])
def previous_by_flight_id(track_id):
    try:
        file_path = Datastore.get_archive_files()[track_id]["path"]
        with open(file_path) as json_file:
            return json.dumps(json.load(json_file))
    except KeyError:
        return make_response("File not Found", 400)


@app.route("/keys", methods=["GET"])
def keys():
    return json.dumps(list(Datastore.keys()))


@app.route("/key/<key>", methods=["GET"])
def get_key(key):
    return json.dumps(Datastore.get_key(key))


@app.route("/location")
def current_location():
    """
    Return the current location as a lat,lon tuple
    """
    location = Datastore.location
    if not location:
        return json.dumps(False)
    x, y = location.location
    return json.dumps(dict(location=[y, x], time=location.timestamp))


@app.route("/bounds")
def bounds():
    """
    Return the current location as a lat,lon tuple
    """
    data = Datastore.bounds
    return json.dumps(
        dict(
            right=data.right, bottom=data.bottom, left=data.left, top=data.top
        )
    )


@app.route("/track")
def track():
    """
    Return the track stored in the Datastore
    """
    return json.dumps([(p.location.y, p.location.x) for p in Datastore.track])


@app.route("/status")
def status():
    """
    Return the track stored in the Datastore
    """
    return json.dumps(
        [
            Datastore.completion("idm"),
            Datastore.completion("scm"),
            {"message": "Uplink: Down", "class": "fail"},
        ]
    )


@app.route("/remaining/<int:z>/<int:x>/<int:y>.png")
def remaining(z, x, y):
    """
    Provide tiles for the remaining heatmap. Generate update tiles if required
    """
    pass


@app.route("/remaining/legend.png")
def remaining_legend():
    """
    Provide a legend for the remaining heatmap.
    """
    pass


@app.route("/completion/<int:z>/<int:x>/<int:y>.png")
def completion(z, x, y):
    """
    Provide tiles for the completion heatmap. Generate update tiles if required
    """
    pass


@app.route("/completion/legend.png")
def completion_legend():
    """
    Provide a legend for the remaining heatmap.
    """
    pass
