import logging
from threading import Condition, Event, RLock, Thread


class GPSWatcher(Thread):
    logger = logging.getLogger("cleargrid.lib.gps.GPSWatcher")

    def __init__(self, client, view=False, min_mode=0):
        super().__init__(daemon=True)
        self.client = client
        self.value = None
        self.update = Condition()
        self.terminate = Event()
        self._callback = None

    @property
    def stream(self):
        while not self.terminate.wait(timeout=1.0):
            with self.update:
                self.update.wait()
                yield self.value

    def update_value(self, value):
        with self.update:
            if self._callback:
                self.logger.debug("Calling update callback")
                self._callback(value)
            self.logger.debug("Received update %r", value)
            self.value = value
            self.update.notify_all()

    def setup(self):
        pass

    def is_valid(self, data):
        return True

    def teardown(self):
        pass

    def run(self):
        with self.client.lock:
            try:
                self.setup()
                while not self.terminate.is_set():
                    data = self.client.receive()[0]
                    if not self.is_valid(data):
                        continue
                    if self.value.__class__ != data.__class__:
                        self.update_value(data)
                    elif data.timestamp > self.value.timestamp:
                        self.update_value(data)
                    elif getattr(data, "mode", 0) > getattr(
                        self.value, "mode", 0
                    ):
                        self.update_value(data)

            except Exception as exc:
                self.logger.exception("Unexpected error")
                self.stop_watch()
                raise exc
            finally:
                self.teardown()

    def stop_watch(self):
        self.terminate.set()
        with self.update:
            self.update.notify_all()

    def start_watch(self, callback=None):
        self.logger.debug("Setting callback to %r", callback)
        self._callback = callback
        self.start()

    def __enter__(self):
        self.start_watch()
        return self

    def __exit__(self, *args):
        self.stop_watch()


class GPSClient(object):
    logger = logging.getLogger("cleargrid.lib.gps.GPSClient")
    host = "127.0.0.1"
    port = None
    watcher_class = GPSWatcher

    def __init__(self, host=None, port=None, **kwargs):
        if host is None:
            host = self.__class__.host
        if port is None:
            port = self.__class__.port
        self.host = host
        self.port = port
        self.lock = RLock()

    @property
    def address(self):
        return "{self.host}:{self.port}".format(self=self)

    def receive(self):
        raise NotImplementedError()

    def watch(self, **kwargs):
        return self.watcher_class(self, **kwargs)
