import logging
import arrow

from pysnmp.hlapi import (
    CommunityData,
    ContextData,
    ObjectIdentity,
    ObjectType,
    SnmpEngine,
    UdpTransportTarget,
    bulkCmd,
)

from ..client import GPSClient
from ..objects import Position
from .exceptions import GPSException

# use specific flags or 'all' for full debugging
# debug.setLogger(debug.Debug('all'))


# iso.org.dod.internet.private.enterprise.cypress.gps.info
BASE_OID = "1.3.6.1.4.1.28952.6.1"
XLAT = {
    f"{BASE_OID}.1.0": "valid",
    f"{BASE_OID}.2.0": "latitude",
    f"{BASE_OID}.3.0": "longitude",
    f"{BASE_OID}.4.0": "altitude",
}


class GPSReport(object):
    def __init__(self, position, valid, timestamp=None):
        if timestamp is None:
            timestamp = arrow.utcnow()
        self.timestamp = timestamp
        self.position = position
        self._valid = None
        self.valid = valid

    @property
    def valid(self):
        return self._valid

    @valid.setter
    def valid(self, value):
        try:
            self._valid = int.from_bytes(value, "little")
        except TypeError:
            self._valid = int(value)


def gps_snmp_iterator(host, port):
    return bulkCmd(
        SnmpEngine(),
        CommunityData("public", mpModel=0),
        UdpTransportTarget((host, port), timeout=1, retries=1),
        ContextData(),
        0,
        4,
        ObjectType(ObjectIdentity(BASE_OID)),
        lookupMib=False,
        lexicographicMode=False,
    )


class CypressGPSClient(GPSClient):
    host = "192.168.1.1"
    port = 161
    logger = logging.getLogger("cleargrid.lib.gps.cypress.CypressGPSClient")

    def __init__(self, host=None, port=None, **kwargs):
        super().__init__(host=host, port=port, **kwargs)
        self.receive()

    def version(self):
        return "Cypress SNMP GPS Client V 0.1"

    def is_valid(self, data):
        return hasattr(data, "valid") and data.valid

    def receive(self):
        data = {
            "latitude": None,
            "longitude": None,
            "altitude": None,
            "valid": 0,
        }
        for (
            errorIndication,
            errorStatus,
            errorIndex,
            varBinds,
        ) in gps_snmp_iterator(self.host, self.port):

            if errorIndication:
                raise GPSException(errorIndication)
            elif errorStatus:
                msg = "{0:s} at {1:s}".format(
                    errorStatus.prettyPrint(),
                    errorIndex and varBinds[int(errorIndex) - 1][0] or "?",
                )
                raise GPSException(msg)
            else:
                for oid, value in varBinds:
                    k = XLAT.get(oid.prettyPrint(), None)
                    if k in data:
                        try:
                            data[k] = float(value.prettyPrint())
                        except ValueError:
                            continue
        valid = data.pop("valid")
        return [
            GPSReport(Position(**data), valid),
        ]


def client_factory(host):
    return type(
        f"{CypressGPSClient.__name__}({host})",
        (CypressGPSClient,),
        {host: host},
    )
