"""Magiccarpet logging extras"""

import json
import logging
import math
import signal
from typing import List, Set

from google.cloud.logging.handlers import ContainerEngineHandler
from cleargrid.logging.handlers.signalsafe import SignalSafeEmitter


# https://gist.github.com/micimize/90c8dc9426f567b5ed7f26b5018c9ffe#file-structured_container_log_handler-py
class StructuredStackdriverHandler(SignalSafeEmitter, ContainerEngineHandler):
    """Extend the ContainerEngineHandler to add additional fields
    """

    additional_fields: List[str]

    def __init__(
        self,
        name: str,
        additional_fields: List[str],
        signals: Set[signal.Signals],
    ):
        super().__init__(signals=signals, name=name)
        self.additional_fields = additional_fields

    def _format_stackdriver_json(self, record, message):
        """Helper to format a LogRecord in in Stackdriver fluentd format.

            :rtype: str
            :returns: JSON str to be written to the log file.
        """
        subsecond, second = math.modf(record.created)

        payload = {
            "message": message,
            "timestamp": {
                "seconds": int(second),
                "nanos": int(subsecond * 1e9),
            },
            "thread": record.thread,
            "severity": record.levelname,
        }

        for field in self.additional_fields:
            value = record.__dict__.get(field, None)
            if value:
                payload[field] = value

        return json.dumps(payload)

    def format(self, record: logging.LogRecord):
        # Yes, skip over direct parents implementation because
        # it is the reason we need to do this ourselves
        message = super(ContainerEngineHandler, self).format(record)
        return self._format_stackdriver_json(record, message)
