import logging
import os

import matplotlib.pyplot as plt
import tilemapbase

tilemapbase.start_logging()
tilemapbase.init(create=True)
OSM = tilemapbase.tiles.build_OSM()

logging.getLogger("PIL.PngImagePlugin").setLevel(logging.INFO)


class Chart(object):
    def __init__(
        self,
        fig=None,
        ax=None,
        figsize=None,
        dpi=100,
        tiles=OSM,
        adjust="box",
        zoom=None,
        **kwargs,
    ):
        self.fig = fig
        self.ax = ax
        self.figsize = (8, 8) if figsize is None else figsize
        self.dpi = dpi
        self.tiles = tiles
        self.adjust = adjust
        self.kwargs = kwargs
        self.zoom = zoom

    def __enter__(self):
        if self.ax is None:
            if self.fig is None:
                self.fig = plt.figure(
                    figsize=self.figsize, dpi=self.dpi, **self.kwargs
                )
            self.ax = self.fig.add_subplot(111)
        elif self.fig is None:
            self.fig = self.ax.get_figure()

        self.ax.set_aspect("equal", self.adjust)
        return self.ax

    def __exit__(self, *args):
        left, right = self.ax.get_xlim()
        bottom, top = self.ax.get_ylim()
        extent = tilemapbase.Extent.from_3857(left, right, top, bottom)
        plotter_args = {}
        if self.zoom:
            plotter_args.update(zoom=self.zoom)
        else:
            plotter_args.update(width=self.figsize[0] * self.dpi)
        plotter = tilemapbase.Plotter(extent, self.tiles, **plotter_args)
        plotter.plot(self.ax, self.tiles)

    def show(self):
        self.fig.show()

    def save(self, path, ext="png", close=True, verbose=True):
        """Save a figure from pyplot.

        Parameters
        ----------
        path : string
            The path (and filename, without the extension) to save the
            figure to.

        ext : string (default='png')
            The file extension. This must be supported by the active
            matplotlib backend (see matplotlib.backends module).  Most
            backends support 'png', 'pdf', 'ps', 'eps', and 'svg'.

        close : boolean (default=True)
            Whether to close the figure after saving.  If you want to save
            the figure multiple times (e.g., to multiple formats), you
            should NOT close it in between saves or you will have to
            re-plot it.

        verbose : boolean (default=True)
            Whether to print information about when and where the image
            has been saved.

        """

        # Extract the directory and filename from the given path
        directory = os.path.split(path)[0]
        filename = "%s.%s" % (os.path.split(path)[1], ext)
        if directory == "":
            directory = "."

        # If the directory does not exist, create it
        if not os.path.exists(directory):
            os.makedirs(directory)

        # The final path to save to
        savepath = os.path.join(directory, filename)

        if verbose:
            print("Saving figure to '%s'..." % savepath)

        # Actually save the figure
        self.fig.savefig(
            savepath, bbox_inches="tight", pad_inches=0, frameon=None
        )

        # Close it
        if close:
            self.close()

        if verbose:
            print("Done")

    def close(self):
        plt.close(self.fig)
