import logging
import pathlib

import appdirs
import arrow
from cleargrid.logging.handlers.on_run_rotation import OnRunRotatingFileHandler

PROJECT_NAME = None

__all__ = ["set_project_name", "add_log_file"]

FORMAT = (
    "%(asctime)s - %(filename)12s:%(lineno)4d - %(threadName)10s"
    "[%(levelname)7s]:%(message)s"
)

_ogger = logging.getLogger("cleargrid.logging.userapp")


def set_project_name(project_name: str, force: bool = False):
    global PROJECT_NAME
    if PROJECT_NAME:
        _ogger.warning("'PROJECT_NAME' has already been set")
        if force:
            _ogger.warning("Forcing redefinition from")
            _ogger.warning("%r", PROJECT_NAME)
            _ogger.warning("to new value")
            _ogger.warning("%r", project_name)

    if not PROJECT_NAME or force:
        PROJECT_NAME = project_name


def add_log_file(log_file: str, project_name: str = None):
    if project_name:
        set_project_name(project_name)
    if not PROJECT_NAME:
        msg = "'PROJECT_NAME' must be set or by provided by a kwarg"
        _ogger.error(msg)
        raise ValueError(msg)
    log_file = {
        "timestamp": arrow.get().isoformat(),
        "default": PROJECT_NAME,
        "true": PROJECT_NAME,
        "false": None,
    }.get(str(log_file).lower(), log_file)

    if not log_file:
        _ogger.debug("No Filehandler added")
        return
    if not log_file.endswith(".log"):
        log_file = f"{log_file}.log"

    log_dir = appdirs.user_log_dir(PROJECT_NAME, "cleargrid")
    log_path = pathlib.Path(log_dir)
    log_path.mkdir(parents=True, exist_ok=True)
    log_path /= log_file

    formatter = logging.Formatter(FORMAT)
    handler = OnRunRotatingFileHandler(log_path, 14)
    handler.setFormatter(formatter)
    logging.getLogger().addHandler(handler)

    _ogger.debug("Added File Handler %r", handler)
