import signal
import threading
from contextlib import contextmanager
from typing import Dict, Collection

_scorebook: Dict[signal.Signals, int] = dict()
_scorebook_lock = threading.Lock()


def _mask_signals(sigs: Collection[signal.Signals]):
    """Mask specified signals.  If called recursively or called
    while signals are already masked, it will do the correct thing
    and not unmask the signals until the corrisponding number of
    exits occur"""
    with _scorebook_lock:
        for sig in sigs:
            count = _scorebook.get(sig, 0)
            # We are using assert to guard against programming errors
            # because logging will call this function when it logs
            # resulting in a deadlock
            assert count >= 0, f"Count for {sig!r} is {count}"
            if count == 0:
                signal.pthread_sigmask(signal.SIG_BLOCK, {sig})
            _scorebook[sig] = count + 1
    # Release the lock before we allow other code to run
    yield

    with _scorebook_lock:
        for sig in sigs:
            count = _scorebook[sig]
            assert count > 0, f"Count for {sig!r} is {count}"
            if count == 1:
                signal.pthread_sigmask(signal.SIG_UNBLOCK, {sig})
            _scorebook[sig] = count - 1


@contextmanager
def mask_signal(sig: signal.Signals):
    return _mask_signals({sig})


@contextmanager
def mask_signals(sigs: Collection[signal.Signals]):
    return _mask_signals(sigs)


def _check_signals(sigs: Collection[signal.Signals]):
    for sig in sigs:
        if sig in signal.sigpending():
            old = signal.pthread_sigmask(signal.SIG_UNBLOCK, {sig})
            # reapply the mask if after allowing the signals we
            # are still here. That is, the handler didn't raise.
            # if so why is this in place????
            signal.pthread_sigmask(signal.SIG_SETMASK, old)


def check_signal(sig: signal.Signals):
    return _check_signals({sig})


def check_signals(sigs: Collection[signal.Signals]):
    return _check_signals(sigs)
