import logging
import random
import time

import requests
from django.conf import settings

logger = logging.getLogger(__name__)


def meter_state(meter):
    messages = ["realtime"]
    if "logging" in meter:
        messages.append("logging")

    if all([meter.get(m, []) for m in messages]):
        return "complete"
    elif any([meter.get(m, []) for m in messages]):
        return "heard"
    return "unheard"


def encode_meter(meter):
    if not meter["longitude"] or not meter["latitude"]:
        return None
    state = meter_state((meter))
    if state == "complete":
        return None
    return {
        "type": "Feature",
        "properties": {"state": state},
        "geometry": {"type": "Point", "coordinates": [meter["longitude"], meter["latitude"]]},
    }


def build_geojson(key, data):
    return {
        key: {
            "type": "FeatureCollection",
            "features": [
                m for m in (encode_meter(meter) for meter in data.values()) if m is not None
            ],
        }
    }


def controller_meter(meter):
    return dict(
        ert=meter["identifier"],
        location=(meter["longitude"], meter["latitude"]),
        idm="logging" in meter,
    )


class Updater:
    def __init__(self, api_server=None):
        """
        Send the endpoints/meters to the front-end. Endpoints was found to be more reliable as it had lat/lng, and is associated 1-1 with a meter
        """
        if api_server is None:
            api_server = settings.API_SERVER
        self.api_server = api_server

    def _build_url(self, endpoint):
        return f"{self.api_server}/{endpoint}"

    def post(self, *args, **kwargs):
        try:
            result = requests.post(*args, **kwargs)
            result.raise_for_status()
        except Exception as e:
            logger.exception("Error communicating with server")

    def reset(self):
        self.post(self._build_url("reset"))

    def remove_meters(self, meter_ids):
        self.post(self._build_url("remove"), json=meter_ids)

    def add_meters(self, meters):
        self.post(
            self._build_url("update"),
            json={"meters": [controller_meter(meter) for meter in meters.values()]},
        )

    def update_readings(self, readings):
        self.post(self._build_url("update"), json={"readings": readings})

    def update_map(self, data):
        geojson = build_geojson("MC3 Integration", data)
        if geojson["MC3 Integration"]["features"]:
            self.post(self._build_url("update"), json=geojson)
