import logging
import threading
import time

from django.conf import settings
from django.db.models import Count

from .data.models import (
    Endpoint,
    Endpointlocation,
    Extendedreading,
    Meter,
    Reading,
    Extendeddlreading,
    Extendeddlrequest,
)
from collections import namedtuple

logger = logging.getLogger(__name__)

Row = namedtuple("Row", ["identifier", "timeoccurred"])


class ORMFetcher:
    def __init__(self):
        self.last_realtime_fetch = None
        self.last_logging_fetch = None
        self.count = None

    @property
    def meter_qs(self):
        return Endpoint.objects.filter(meter__isoutofroute=False)

    @property
    def meters_updated(self):
        count = self.meter_qs.count()
        updated = count != self.count
        self.count = count
        return updated

    @property
    def realtime_messages(self):
        f = {"meter__isoutofroute": False}
        if self.last_realtime_fetch is not None:
            f.update(timeoccurred__gt=self.last_realtime_fetch)

        readings = [
            Row(identifier=r.meter__identifier, timeoccurred=r.timeoccurred)
            for r in Reading.objects.filter(**f).values_list(
                "meter__identifier", "timeoccurred", named=True
            )
        ]
        self.last_realtime_fetch = max(
            [r.timeoccurred for r in readings], default=self.last_realtime_fetch
        )
        return readings

    @property
    def logging_messages(self):
        f = {}
        if self.last_logging_fetch is not None:
            f.update(timeoccurred__gt=self.last_logging_fetch)

        # Because there is no relation between the Extendeddlreading
        # and the Extendeddlrequest tables, we need to query them
        # individually and then merge the results
        readings = {
            r.extdlreq: r
            for r in Extendeddlreading.objects.filter(**f)
            .values_list("extdlreq", "timeoccurred", named=True)
            .order_by("timeoccurred")
        }
        requests = {
            r.PK: r
            for r in Extendeddlrequest.objects.filter(
                PK__in=list(readings.keys())[:500], meter__isoutofroute=False
            ).values_list("PK", "meter__identifier", named=True)
        }

        # merge the two into a standardized reading report as from the realtime messages
        readings = [
            Row(identifier=requests[k].meter__identifier, timeoccurred=readings[k].timeoccurred)
            for k in requests
        ]
        self.last_logging_fetch = max(
            [r.timeoccurred for r in readings], default=self.last_logging_fetch
        )
        return readings

    @property
    def meters(self):
        return self.meter_qs.annotate(dl_requests=Count("meter__extended_dl_requests")).values_list(
            "identifier", "latitude", "longitude", "dl_requests", named=True
        )


class PollingThread(threading.Thread):
    """
    Arguments: 
        polling_interval (int): Seconds for every poll
        result_q (queue): Simple Python FIFO queue
        fetcher (object): a class to return normalized results from the datastore
    """

    def __init__(self, polling_interval, result_q, *args, fetcher=None, **kwargs):
        super().__init__(*args, **kwargs)
        # A flag to notify the thread that it should finish up and exit
        self._exit = threading.Event()
        if fetcher is None:
            fetcher = ORMFetcher()
        self.fetcher = fetcher
        self.result_q = result_q
        if polling_interval is None:
            polling_interval = settings.POLLING_INTERVAL
        self.polling_interval = polling_interval
        self._data = {}
        self._data_lock = threading.Lock()

    @property
    def data(self):
        with self._data_lock:
            return dict(self._data)

    @staticmethod
    def build_meter(meter):
        d = dict(
            identifier=meter.identifier,
            latitude=meter.latitude,
            longitude=meter.longitude,
            realtime=[],
        )
        if meter.dl_requests > 0:
            d.update(logging=[])
        return d

    def queue_removed_meters(self, ids):
        removed = set(self._data.keys()) - ids
        with self._data_lock:
            for k in removed:
                del self._data[k]
        if removed:
            self.result_q.put(("remove_meters", list(removed)))
            logger.debug("Removed %d meters")
        return removed

    def queue_added_meters(self, ids, meters):
        added = ids - set(self._data.keys())
        new_meters = {
            meter.identifier: self.build_meter(meter)
            for meter in (m for m in meters if m.identifier in added)
        }
        with self._data_lock:
            self._data.update(new_meters)
        if new_meters:
            self.result_q.put(("add_meters", new_meters))
            logger.debug("Added %d new meters", len(new_meters))
        return new_meters

    def _find_messages_updated(self, message_type, messages):
        '''RETURNS (dict) updates'''
        updates = {}
        with self._data_lock:
            for m in messages:
                try:
                    meter = self._data[m.identifier]
                except:
                    logging.error("Unable to find record for meter %r", m.identifier)
                    continue
                updates.setdefault(m.identifier, []).append(
                    ("scm" if message_type == "realtime" else "idm", m.timeoccurred.timestamp())
                )
                meter[message_type].append(updates[m.identifier][-1])
        return updates

    def _queue_meters(self) -> int:
        # see if our meterset has changed
        try:
            if self.fetcher.meters_updated:
                meters = self.fetcher.meters
                ids = set([m.identifier for m in meters])
                removed = self.queue_removed_meters(ids)
                new_meters = self.queue_added_meters(ids,meters)
                return len(removed) + len(new_meters)
                
        except Exception:
            logger.exception("Error while processing updates to meter set")
        return 0

    def _queue_realtime_messages(self) -> int:
        # see if we have new realtime (SCM) messages
        try:
            updates = self._find_messages_updated("realtime", self.fetcher.realtime_messages)
            if updates:
                self.result_q.put(("update_readings", updates))
                logging.debug("Received %d new %s messages", len(updates), "realtime")
            return len(updates)
        except Exception:
            logger.exception("Error while checking for new realtime messages")
        return 0

    def _queue_logging_messages(self) -> int:
        # see if we have new logging (SCM) messages
        try:
            updates = self._find_messages_updated("logging", self.fetcher.logging_messages)
            if updates:
                self.result_q.put(("update_readings", updates))
                logging.debug("Received %d new %s messages", len(updates), "logging")
            return len(updates)
        except Exception:
            logger.exception("Error while checking for new logging messages")
        return 0

    def process(self):
        start = time.time()
        if self._queue_meters() + self._queue_realtime_messages() + self._queue_logging_messages() > 0:
            self.result_q.put(("update_map", dict(self.data)))
            logger.debug("Requesting Map update")
        else:
            logger.debug("No updates found in database")

        elapsed = time.time() - start
        logger.debug("completed processing in %0.5f seconds", elapsed)
        return max([0, self.polling_interval - elapsed])

    def run(self):
        wait = self.process()
        while not self._exit.wait(timeout=self.polling_interval):
            wait = self.process()

    def join(self, timeout=None):
        self._exit.set()
        super().join(timeout)
