import argparse
import logging
import struct

from cleargrid.r900.r900 import DataStream

SAMPLE_RATE = 2359296


def parse_args():
    parser = argparse.ArgumentParser(
        description="Extract Neptune R900 readings from RF input"
    )

    parser.add_argument(
        "source",
        type=str,
        default="-",
        help="Input source (Default: <stdin>)",
    )
    parser.add_argument(
        "--input-filter",
        "--if",
        type=str,
        default="cleargrid.r900.input_filters.source_detect",
        help="Input filter to use when reading file",
    )
    parser.add_argument(
        "--samplerate",
        type=int,
        default=SAMPLE_RATE,
        help="Sample rate of input",
    )
    parser.add_argument(
        "--blocksize",
        type=int,
        default=SAMPLE_RATE // 10,
        help="Number of samples to load and process at a time",
    )

    parser.add_argument(
        "--output",
        type=argparse.FileType("wb"),
        default="-",
        help="Output File (Default: <stdout>)",
    )

    parser.add_argument(
        "--offset",
        type=int,
        default=0,
        help="Number of samples to seek into the input before processing",
    )
    parser.add_argument(
        "--sample-count",
        type=int,
        default=None,
        help="Number of samples to process",
    )
    parser.add_argument(
        "--time-offset",
        type=int,
        default=None,
        help="Number of  to seek into the input before processing",
    )
    parser.add_argument(
        "--duration",
        type=int,
        default=None,
        help="Microseconds of data to process",
    )
    parser.add_argument(
        "--log-level",
        "-L",
        type=str,
        default="info",
        help="Level to log messages at",
    )

    args = parser.parse_args()
    if args.time_offset:
        args.offset = args.time_offset / 1e6 * args.samplerate
    if args.duration:
        args.sample_count = np.ceil(args.duration / 1e6 * args.sample_rate)

    return args


def main():
    args = parse_args()
    logging.basicConfig(level=getattr(logging, args.log_level.upper()))
    logging.debug(args)
    logging.getLogger("matplotlib").setLevel(logging.ERROR)
    logging.getLogger("numba").setLevel(logging.WARNING)
    ds = DataStream(
        source=args.source,
        filter_spec=args.input_filter,
        sample_rate=args.samplerate,
        start_sample=args.offset,
        blocksize=args.blocksize,
    )
    preamble = "111145"
    preamble = None
    seen = set()
    packet_format = ">I6s"
    packet_size = struct.calcsize(packet_format)
    for offset, match, candidate in ds.process(preamble):
        if len(candidate) < packet_size:
            continue
        try:
            candidate = candidate[:packet_size]
            if candidate in seen:
                continue
            else:
                seen.add(candidate)
            serial, payload = struct.unpack(packet_format, candidate)
            data = dict(
                unk1=payload[0],
                nouse=(payload[1] & 0xFC) >> 2,
                backflow=payload[1] & 0x03,
                consumption=(
                    (payload[2] << 16) | (payload[3] << 8) | payload[4]
                ),
                unk3=(payload[5] & 0xC0) >> 6,
                leak=(payload[5] & 0x3C) >> 2,
                leaknow=payload[5] & 0x03,
            )
            logging.debug("%r\t%r", offset, ds.timestamp(offset))
            logging.debug(serial)
            logging.debug(data)
            data_str = ", ".join(
                [
                    f"{k}: {v:{w}d}"
                    for w, (k, v) in zip([3, 2, 1, 8, 1, 2, 1], data.items())
                ]
            )
            print(f"{offset:10d} {serial:8d} {{{data_str}}}")
        except Exception:
            logging.exception(
                "error processing candidate %r at offset %r", candidate, offset
            )


if __name__ == "__main__":
    main()
