import logging
import os
import signal
import typing as _t

import configargparse
import sentry_sdk
from sentry_sdk.integrations.logging import LoggingIntegration

from ._logging import (
    configure_logging,
    help_handlers,
    parse_log_handlers,
    parse_log_level,
)
from .datastructures import Sentry


def fly_args(argparser: configargparse.ArgParser) -> configargparse.ArgParser:
    argparser.add_argument(
        "--log-level",
        type=parse_log_level,
        help="Root Loglevel",
        env_var="LOG_LEVEL",
        default="INFO",
    )
    argparser.add_argument(
        "--log-handler",
        type=parse_log_handlers,
        help=help_handlers,
        env_var="LOG_HANDLER",
        default="console",
    )
    # TODO: configure loglevels for individual loglevels
    return argparser


def find_environment():
    namespace = os.environ.get("POD_NAMESPACE", "Dev")
    if namespace.lower() in ("production", "staging", "dev"):
        return namespace.title()
    else:
        return "Review"


ALWAYS_MASKED_SIGNALS: _t.Set[signal.Signals] = set(
    [
        # Ctrl^C on the keyboard.  The user is requesting we
        # exit, these should not be eaten by the logger.
        signal.SIGINT,
        # System is requesting we exit, these should not be
        # eaten by the logger.
        signal.SIGTERM]
)


class Fly:
    """Initialize a server app, logging, sentry, etc."""

    def __init__(
        self,
        *,
        arg_parser: configargparse.ArgParser,
        sentry: Sentry,
        main_class: _t.Optional[_t.Callable] = None,
        main_func: _t.Optional[_t.Callable] = None,
        masked_signals: _t.Optional[_t.Set[signal.Signals]] = None,
    ):
        self.arg_parser = arg_parser
        self.sentry = sentry

        if main_class and not main_func:
            self.main_class: _t.Optional[_t.Callable] = main_class
            self.main_func: _t.Optional[_t.Callable] = None
        elif main_func and not main_class:
            self.main_class = None
            self.main_func = main_func
        else:
            raise ValueError(
                f"Exactly one of main_class ({main_class!r}) "
                f"or main_func ({main_func!r}) must be set"
            )
        if masked_signals:
            self.masked_signals = masked_signals | ALWAYS_MASKED_SIGNALS
        else:
            self.masked_signals = ALWAYS_MASKED_SIGNALS

    def __call__(self):
        args = self.arg_parser.parse_args()

        sentry_integrations = (
            set(
                [
                    LoggingIntegration(
                        # Capture info and above as breadcrumbs
                        level=logging.INFO,
                        # Do not send errors as events, only critical is
                        # worthy of sending to Sentry
                        event_level=logging.CRITICAL,
                    )
                ]
            )
            | self.sentry.additional_integrations
        )

        sentry_sdk.init(
            dsn=self.sentry.dsn,
            release=self.sentry.release,
            environment=find_environment(),
            integrations=list(sentry_integrations),
        )

        print(args)
        configure_logging(
            self.masked_signals, args.log_level, args.log_handler
        )

        if self.main_class:
            main = self.main_class(args)
            return main.run()
        elif self.main_func:
            return self.main_func(args)
        else:
            raise NotImplementedError()
