import struct
from commlib.proto.message_pb2 import Message
from commlib.UDP.payloadconstructor import get_unix_time, PayloadConstructor
from commlib.crypto_lob import (
    create_sha2_signature,
    create_sha3_signature,
    create_signing_string,
    tskdf,
    Encryptor,
)


def construct_message(collector, sequence, cipher, mac):
    """
    ARGS:
        payload_type: One of ping | location_message | location_sync_request 
        | location_sync_message | mission_status_sync_message | 
        mission_status_sync_request | mission_status_update_message | 
        prepare_mission_message | complete_mission_message | mission_state_message
        cipher: Message.AES_256_CTR_TSKDF_HMAC_SHA2_512 | Message.AES_256_CTR_TSKDF_HMAC_SHA3_512
        mac: Message.HMAC_SHA2_256_KDF_HMAC_SHA2_512 | Message.HMAC_SHA3_256_KDF_HMAC_SHA3_512
    """
    message = Message()
    message.cipher = cipher
    message.mac = mac
    message.collector = collector
    if sequence is not None:
        message.sequence = sequence
    message.currenttime = get_unix_time()
    return message


def encrypt_and_sign_message(
    tskdf, constructed_message, payload, encryptor, create_signature, cipher, mac
):
    """
    Encrypts and signs the constructed_message and payload
    ARGS: 
        tskdf
        constructed_message
        payload (Payload | None)
        encryptor
        create_signature
        cipher
        mac
    RETURNS: 
        data (string): Serialized Message
        cipher: Message.AES_256_CTR_TSKDF_HMAC_SHA2_512 | Message.AES_256_CTR_TSKDF_HMAC_SHA3_512
        mac: Message.HMAC_SHA2_256_KDF_HMAC_SHA2_512 | Message.HMAC_SHA3_256_KDF_HMAC_SHA3_512
    """
    # Serialize the payload
    if payload:
        stringified_payload = payload.SerializeToString()
    if tskdf is None:
        constructed_message.payload = stringified_payload
        data = constructed_message.SerializeToString()
    else:
        # Manage encryption
        timestep = tskdf.derive_timestep()

        # Produce the Encryption Key
        kdf_key, kdf_iv = tskdf.derive(timestep, constructed_message.sequence)

        if payload:
            encrypted_payload = encryptor.encrypt_payload(
                kdf_key, kdf_iv, stringified_payload
            )
            constructed_message.payload = encrypted_payload

        # Manage Signing
        signing_string = create_signing_string(constructed_message)
        derived_key = tskdf.derive_hmac()
        constructed_message.signature = bytes(
            create_signature(derived_key, signing_string), "utf-8"
        )
        data = constructed_message.SerializeToString()

    return data
