import logging
import time

import configargparse
import requests
import requests.exceptions

from cleargrid.gps import get_client
from cleargrid.logging.userapp import add_log_file


def send_to_interface(update, retries=5):
    attempt = retries
    while True:
        try:
            requests.post(
                "http://127.0.0.1/update",
                json={
                    "location": (
                        update.timestamp.timestamp,
                        update.position.latitude,
                        update.position.longitude,
                    )
                },
            )
            break
        except requests.exceptions.ConnectionError as e:
            if not attempt:
                raise e
            timeout = 2 ** (retries - attempt)
            logging.warn(
                "Connection to interface host failed, retrying in %d seconds",
                timeout,
            )
            time.sleep(timeout)
            attempt -= 1


def watch(w, retries=5):
    for i, p in enumerate(w.stream):
        if hasattr(p, "position"):
            send_to_interface(p, retries)


def main():
    p = configargparse.get_arg_parser(
        default_config_files=["/etc/cleargrid", "~/.cleargrid"]
    )
    p.add("-c", "--config", is_config_file=True, help="config file path")
    p.add_argument("--log-level", "-L", default="DEBUG", env_var="LOG_LEVEL")
    p.add_argument("--log-file", type=str, default=True, env_var="LOG_FILE")
    args = p.parse_known_args()[0]

    logging.basicConfig(level=getattr(logging, args.log_level.upper()))
    add_log_file(args.log_file, "gps_client")

    while True:
        c = get_client()
        logging.info(c.version())
        with c.watch() as w:
            try:
                watch(w)
            except requests.exceptions.ConnectionError:
                logging.error("Unable to connect to interface host")
            except Exception:
                logging.exception("Error in Watcher, retrying")


if __name__ == "__main__":
    main()
